#region References

using System;
using System.Data;
using System.Collections;
using System.Text.RegularExpressions;
using System.Text;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;
using Method = System.Reflection.MethodBase;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

#endregion

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Gregory Lohse</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>10/20/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>BloodUnit class</summary>

	#endregion

	public class BloodUnit : gov.va.med.vbecs.BOL.BloodProduct, IRecordStatusCode
	{

		#region Variables

		/// <summary>
		/// Indicator that will be set in the bloodunit constructor, so we don't build multiple bloodproduct 
		/// </summary>
		private bool _needsBloodProduct = false;

		/// <summary>
		/// _bloodUnitDataTable
		/// </summary>
		private DataTable _bloodUnitDataTable;
        
		/// <summary>
		/// _bloodUnitDataRow
		/// </summary>
		private DataRow _bloodUnitDataRow;
        
		/// <summary>
		/// _bloodUnitMedia
		/// </summary>
		private BloodUnitMedia _bloodUnitMedia;
        
		/// <summary>
		/// _bloodUnitFinance
		/// </summary>
		private BloodUnitFinance _bloodUnitFinance;
        
		/// <summary>
		/// _bloodUnitStatus
		/// </summary>
		private BloodUnitStatus _bloodUnitStatus;

		private Common.BarcodeType _barcodeType;
		
		/// <summary>
		/// _isbtUnitID
		/// </summary>
		private Regex _isbtUnitID;

		private Regex _scannedISBTUnitID;
		
		/// <summary>
		/// _codabarEyeReadableUnitID
		/// </summary>
		private Regex _codabarEyeReadableUnitID;

		/// <summary>
		/// _codabarScannedUnitID
		/// </summary>
		private Regex _codabarScannedUnitID; 
		
		/// <summary>
		/// _assignedToPatients
		/// </summary>
		private System.Collections.ArrayList _assignedToPatients;
        
		/// <summary>
		/// _patientSpecimens
		/// </summary>
		private BOL.Specimen[] _patientSpecimens;
        
		/// <summary>
		/// _directedPatient
		/// </summary>
		private Patient _directedPatient;
        
		/// <summary>
		/// _localSupplier
		/// </summary>
		private BOL.LocalSupplier _localSupplier;

		/// <summary>
		/// _patientAssignments
		/// </summary>
		private System.Data.DataTable _patientAssignments;
                
		/// <summary>
		/// _incomingShipment
		/// </summary>
		private BOL.IncomingShipment _incomingShipment;
        
		/// <summary>
		/// _checkCharacter
		/// </summary>
		private string _checkCharacter;

		/// <summary>
		/// _recordStatusCode reflects the state of the record e.g.  Active, Inactive
		/// </summary>
		private Common.RecordStatusCode _recordStatusCode = Common.RecordStatusCode.Active;

		#endregion
 
		#region Constructors

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5102"> 
		///		<ExpectedInput>eyereadable, productcode, barcodeType, guid</ExpectedInput>
		///		<ExpectedOutput>BloodUnit object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5103"> 
		///		<ExpectedInput>Invalid eyereadable, productcode, barcode type, guid</ExpectedInput>
		///		<ExpectedOutput>new BloodUnit object</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// M.Jablonski - constructor
		/// </summary>
		/// <param name="eyeReadableUnitID"></param>
		/// <param name="productCode"></param>
		/// <param name="barcodeType"></param>
		///<param name="productGuid"></param>
		public BloodUnit(string eyeReadableUnitID, string productCode, Common.BarcodeType barcodeType, System.Guid productGuid): base(productCode, productGuid)
		{
			string errorMessage = string.Empty;
			this._codabarEyeReadableUnitID = Common.RegularExpressions.CodabarUnitID();
			this._codabarScannedUnitID = Common.RegularExpressions.ScannedCodabarUnitID();
			this._isbtUnitID = Common.RegularExpressions.ISBT128UnitID();
			this._scannedISBTUnitID = Common.RegularExpressions.ScannedISBT128UnitID();
			this._barcodeType = barcodeType;
			Regex regexCodabarProductCode = Common.RegularExpressions.CodabarProductCode();
			Regex regexISBT128ProductCode = Common.RegularExpressions.ISBT128ProductCode();

			
			

			if(barcodeType == Common.BarcodeType.ISBT128)
			{
				if(!_isbtUnitID.IsMatch(eyeReadableUnitID))
				{
					throw new BusinessObjectException(Common.StrRes.SysErrMsg.UC001.IncorrectUnitID().ResString);
				}
				if(!regexISBT128ProductCode.IsMatch(productCode))
				{
					throw new BusinessObjectException(Common.StrRes.SysErrMsg.UC001.InvalidProductCode().ResString);
				}
			}
			else
			{
				if(!_codabarEyeReadableUnitID.IsMatch(eyeReadableUnitID))
				{
					throw new BusinessObjectException(string.Concat(Common.StrRes.SysErrMsg.UC001.IncorrectUnitID().ResString));
				}
				if(!regexCodabarProductCode.IsMatch(productCode))
				{
					throw new BusinessObjectException(Common.StrRes.SysErrMsg.UC001.InvalidProductCode().ResString);
				}
			}

			//Getting a BloodUnit by ID/Product requires the FULL productcode, so hopefully the UI
			//has sent it in for ISBT units -- otherwise the call to GetBloodUnitByUnitIDProduct might
			//not find the unit and data errors will occur
			if (barcodeType == Common.BarcodeType.ISBT128)
			{
				//Defaults -- based on UC_1
				this.DonationType = new BOL.DonationType(Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.VolunteerAllogeneicDonor));
				this.UnitDivisionId  = "00";

				if (productCode.Trim().Length == 8)
				{
					this.DonationType = new BOL.DonationType(productCode[5]);
					this.UnitDivisionId = productCode.Substring(6);
				}
				else
				{
					if (productCode.Trim().Length == 6)
					{
						this.DonationType = new BOL.DonationType(productCode[5]);
					}
				}
			}

			_bloodUnitDataTable = DAL.BloodUnit.GetBloodUnitByUnitIdProduct(eyeReadableUnitID, this.FullProductCode, productGuid);
			if (_bloodUnitDataTable.Rows.Count > 0)
			{
				this.IsNew = false;
				if(_bloodUnitDataTable.Columns.Contains(TABLE.BloodUnit.RecordStatusCode))
				{
					_bloodUnitDataRow = _bloodUnitDataTable.Rows[0];
					this.BloodProductGuid = (System.Guid) _bloodUnitDataTable.Rows[0][TABLE.BloodUnit.BloodProductGuid];
				}
			}
			else
			{
				MakeNew();
			}
            
			if (this.IsNew)
			{
				this.BarcodeType = barcodeType;
				
				//07/25/2005---Code change to lighten restriction on UnitID for UC_24
				this.SetEyeReadableUnitId(eyeReadableUnitID, false);
				this.ProductCode = productCode;
				if(this.BarcodeType == Common.BarcodeType.ISBT128)
				{
					_bloodUnitDataRow[TABLE.BloodUnit.LabelUnitId] = eyeReadableUnitID;
				}
			}

			this.InstantiateAggregates(_bloodUnitDataRow);
			this.LoadFromDataRow(_bloodUnitDataRow);
			
			_bloodUnitDataRow.AcceptChanges();
			_bloodUnitDataTable.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
			this.SetEvents();
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5104"> 
		///		<ExpectedInput>bloodunitguid</ExpectedInput>
		///		<ExpectedOutput>Bloodunit object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5105"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>New Bloodunit object</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodUnit Constructor
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public BloodUnit(System.Guid bloodUnitGuid)
		{
			this._codabarEyeReadableUnitID = Common.RegularExpressions.CodabarUnitID();
			this._codabarScannedUnitID = Common.RegularExpressions.ScannedCodabarUnitID();
			this._isbtUnitID = Common.RegularExpressions.ISBT128UnitID();
			this._scannedISBTUnitID = Common.RegularExpressions.ScannedISBT128UnitID();

			_bloodUnitDataTable = DAL.BloodUnit.GetBloodUnitByGuid(bloodUnitGuid);

			if (_bloodUnitDataTable.Rows.Count > 0)
			{
				this.IsNew = false;
				_bloodUnitDataRow = _bloodUnitDataTable.Rows[0];
				this.ProductCode = _bloodUnitDataTable.Rows[0][TABLE.BloodUnit.UnitProductCode].ToString();
				this.RecordStatusCode = Common.Utility.GetRecordStatusCodeFromString(_bloodUnitDataTable.Rows[0][TABLE.BloodUnit.RecordStatusCode].ToString());
				this.BloodProductGuid = (System.Guid) _bloodUnitDataTable.Rows[0][TABLE.BloodUnit.BloodProductGuid];
			}
			else
			{
				MakeNew();
				this.BloodUnitGuid = bloodUnitGuid;
			}
			
			//Since we don't go through the "base" BloodProduct constructor, we want the aggregate built
			this._needsBloodProduct = true;

			this.InstantiateAggregates(_bloodUnitDataRow);
			this.LoadFromDataRow(_bloodUnitDataRow);
			

			_bloodUnitDataRow.AcceptChanges();
			_bloodUnitDataTable.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
			this.SetEvents();
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5106"> 
		///		<ExpectedInput>unitguid, productguid</ExpectedInput>
		///		<ExpectedOutput>BloodUnit object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5107"> 
		///		<ExpectedInput>invalid unitguid, productguid</ExpectedInput>
		///		<ExpectedOutput>New BloodUnit object</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodUnit Constructor
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="bloodProductGuid"></param>
		public BloodUnit(System.Guid bloodUnitGuid, System.Guid bloodProductGuid) : base(bloodProductGuid)
		{
			this._codabarEyeReadableUnitID = Common.RegularExpressions.CodabarUnitID();
			this._codabarScannedUnitID = Common.RegularExpressions.ScannedCodabarUnitID();
			this._isbtUnitID = Common.RegularExpressions.ISBT128UnitID();
			this._scannedISBTUnitID = Common.RegularExpressions.ScannedISBT128UnitID();


			_bloodUnitDataTable = DAL.BloodUnit.GetBloodUnitByGuid(bloodUnitGuid);

			if (_bloodUnitDataTable.Rows.Count > 0)
			{
				this.IsNew = false;
				_bloodUnitDataRow = _bloodUnitDataTable.Rows[0];
				this.BloodProductGuid = (System.Guid) _bloodUnitDataTable.Rows[0][TABLE.BloodUnit.BloodProductGuid];
			}
			else
			{
				MakeNew();
				this.BloodUnitGuid = bloodUnitGuid;
			}
			this.InstantiateAggregates(_bloodUnitDataRow);
			this.LoadFromDataRow(_bloodUnitDataRow);

			_bloodUnitDataRow.AcceptChanges();
			_bloodUnitDataTable.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
			this.SetEvents();
		}

		///<Developers>
		///	<Developer>Oliver Kloozoff</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5108"> 
		///		<ExpectedInput>Datarow</ExpectedInput>
		///		<ExpectedOutput>BloodUnit object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5109"> 
		///		<ExpectedInput>Invalid DataRow</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodUnit Constructor
		/// </summary>
		/// <param name="bloodUnitDataRow"></param>
		public BloodUnit(DataRow bloodUnitDataRow)
		{
			this.IsNew = false;

			this._codabarEyeReadableUnitID = Common.RegularExpressions.CodabarUnitID();
			this._codabarScannedUnitID = Common.RegularExpressions.ScannedCodabarUnitID();
			this._isbtUnitID = Common.RegularExpressions.ISBT128UnitID();
			this._scannedISBTUnitID = Common.RegularExpressions.ScannedISBT128UnitID();

			_bloodUnitDataRow = bloodUnitDataRow;
			this.ProductCode = (string) _bloodUnitDataRow[TABLE.BloodUnit.UnitProductCode];
			this.BloodProductGuid = (System.Guid) _bloodUnitDataRow[TABLE.BloodUnit.BloodProductGuid];
			_bloodUnitDataTable = bloodUnitDataRow.Table;

			//Since we don't go through the "base" BloodProduct constructor, we want the aggregate built
			this._needsBloodProduct = true;
			InstantiateAggregates(bloodUnitDataRow);
			this.LoadFromDataRow(bloodUnitDataRow);

			_bloodUnitDataRow.AcceptChanges();
			_bloodUnitDataTable.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
			this.SetEvents();
		}

		#endregion

		#region Properties
		
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5110"> 
		///		<ExpectedInput>BOL.IncomingShipment</ExpectedInput>
		///		<ExpectedOutput>BOL.IncomingShipment</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5111"> 
		///		<ExpectedInput>Invalid BOL.IncomingShipment</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// IncomingShipment
		/// </summary>
		public BOL.IncomingShipment IncomingShipment
		{
			get 
			{
				if (_incomingShipment == null) 
				{
					_incomingShipment = new BOL.IncomingShipment(this.IncomingShipmentGuid);
				}
				return _incomingShipment;
			}
			set 
			{
				if (this.IncomingShipmentGuid != System.Guid.Empty) 
				{
					if (value.ShipmentGuid != this.IncomingShipmentGuid) 
					{
						throw new ArgumentException("Invalid Argument: mismatch", "value");
					}
				}
				else
				{
					this.IncomingShipmentGuid = value.ShipmentGuid;
				}
				_incomingShipment = value;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/3/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4654"> 
		///		<ExpectedInput>bool, Specimen</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4655"> 
		///		<ExpectedInput>bool, specimen</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean value indicating if the unit is eligible 
		/// for emergency issue.
		/// BR_3.05, BR_32.20, BR_73.06,BR_16.06, TT_52.03
		/// </summary>
		/// CR2899: Pass in specimen information instead of searching for it
		public bool EligibleForEmergencyIssue(bool optionalCrossmatchEligible, bool specimenDiscrepancyExists, bool specimenABORhTested)
		{
			bool eligible = false;
			switch (this.BloodComponentClassId)
			{
				case (int) Common.ComponentClass.WB:
					break;
					// Unit must either be:
					// 1) type O or can be 
					// 2) ABO specific if specimen has been ABORh tested and Rh is Positive or Negative
				case (int) Common.ComponentClass.RBC:
					if ((_bloodUnitMedia.AboRh.RH == Common.RH.Positive ||
						_bloodUnitMedia.AboRh.RH == Common.RH.Negative) &&
						_bloodUnitMedia.AboRh.Abo == Common.ABO.O || 
						(specimenABORhTested && !specimenDiscrepancyExists))
						eligible = true;
					break;
				case (int) Common.ComponentClass.FFP:
					// CR 2567: Deleted the check for Rh
					if (_bloodUnitMedia.AboRh.Abo == Common.ABO.AB || 
						(specimenABORhTested && !specimenDiscrepancyExists))
						eligible = true;
					break;
				case (int) Common.ComponentClass.PLT:
				case (int) Common.ComponentClass.CRYO:
					eligible = true;
					break;
					// BR_3.09
				case (int) Common.ComponentClass.Other:
					if (!optionalCrossmatchEligible) 
					{
						eligible = true;
					}
					else
					{
						if ((_bloodUnitMedia.AboRh.RH == Common.RH.Positive ||
							_bloodUnitMedia.AboRh.RH == Common.RH.Negative) &&
							_bloodUnitMedia.AboRh.Abo == Common.ABO.O || 
							(specimenABORhTested && !specimenDiscrepancyExists))
							eligible = true;
					}
					break;
			}
			return eligible;
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4664"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4665"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Indicates whether or not this unit is eligible for optional crossmatch
		/// BR_3.09
		/// </summary>
		public bool OptionalCrossmatch
		{	
			get
			{
				bool optionalCrossmatch = false;

				if (this.BloodComponentClassId == (int) Common.ComponentClass.Other)
				{
					switch (Common.Utility.GetProductTypeFromProductTypeCode(this.ProductType.ProductTypeCode))
					{
							// Plasma/ Apheresis plasma
						case Common.ProductType.ApheresisPlasma:
						case Common.ProductType.ThawedApheresisPlasma:
						case Common.ProductType.Plasma:
						case Common.ProductType.ThawedPlasma:
							// Platelet rich plasma
						case Common.ProductType.PlateletRichPlasma:
							// Pooled plasma
						case Common.ProductType.PooledPlasma:
						case Common.ProductType.LiquidApheresisPlasma:
							optionalCrossmatch = false;
							break;

							// Granulocytes
						case Common.ProductType.Granulocytes:
						case Common.ProductType.ApheresisGranulocytes:
						case Common.ProductType.PooledGranulocytes:
							// Apheresis granulocytes platelets
						case Common.ProductType.ApheresisGranulocytesPlatelets:
							// Leukocytes
						case Common.ProductType.Leukocytes:
						case Common.ProductType.ApheresisLeukocytes:
							// Platelet rich buffy coat
						case Common.ProductType.PlateletRichBuffyCoat:
						case Common.ProductType.PooledPlateletRichBuffyCoat:
							// Apheresis lymphocytes
						case Common.ProductType.ApheresisLymphocytes:
							// Apheresis monocytes
						case Common.ProductType.ApheresisMonocytes:
							optionalCrossmatch = true;
							break;
					}
				}
				
				return optionalCrossmatch;
			}
		}
        
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5112"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5113"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This is the 8 Character product code (for ISBT units)
		/// For Codabar units it's the same ol' 5 char UnitProductCode
		/// </summary>
		public string FullProductCode
		{
			get
			{
				return GetFullProductCode(this.BarcodeType, this.ProductCode, this.DonationType.DonationTypeCode, this.UnitDivisionId);
			}
		}

		/// <summary>
		/// Gets the full product code.
		/// This is the 8 Character product code (for ISBT units)
		/// For Codabar units it's the same ol' 5 char UnitProductCode
		/// </summary>
		/// <param name="barcodeType"></param>
		/// <param name="unitProductCode"></param>
		/// <param name="donationTypeCode"></param>
		/// <param name="unitDivisionId"></param>
		/// <returns></returns>
		public static string GetFullProductCode(Common.BarcodeType barcodeType, string unitProductCode, char donationTypeCode, string unitDivisionId)
		{
			if (barcodeType == Common.BarcodeType.ISBT128)
			{
				return unitProductCode + donationTypeCode + unitDivisionId;
			}
			else
			{
				return unitProductCode;
			}
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5114"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5115"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodUnitGuid
		/// </summary>
		public System.Guid BloodUnitGuid
		{
			get
			{
				if (_bloodUnitDataRow.IsNull(TABLE.BloodUnit.BloodUnitGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)_bloodUnitDataRow[TABLE.BloodUnit.BloodUnitGuid];
			}
			set
			{
				_bloodUnitDataRow[TABLE.BloodUnit.BloodUnitGuid] = value;
			} 

		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5116"> 
		///		<ExpectedInput>unitid, out string</ExpectedInput>
		///		<ExpectedOutput>out string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5117"> 
		///		<ExpectedInput>Invalid UnitId</ExpectedInput>
		///		<ExpectedOutput>Invalid out string</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///     Validates for //BR_1.05
		///The system must evaluate any manually entered number for rules entered in UC_06 associated with that FDA registration number and collection facility evaluating and allowing entry only when:
		///the entered Unit ID follows the rules as either all numerics or alphanumerics as defined for the collection facility
		///the entered 1.0 matches the previously defined prefix for the collection facility 
		/// </summary>
		/// <param name="unitID"></param>
		/// <param name="validMessage"></param>
		public void CheckValidUnitIDForLocalSupplier(string unitID, out string validMessage)
		{
			validMessage = string.Empty;
			int nCount = 0;
			//BR_1.05
			//The system must evaluate any manually entered number for rules entered in UC_06 associated with that FDA registration number and collection facility evaluating and allowing entry only when:
			//the entered Unit ID follows the rules as either all numerics or alphanumerics as defined for the collection facility
			//the entered 1.0 matches the previously defined prefix for the collection facility 
			if(this._localSupplier != null)
			{
				if(this._localSupplier.FDARegistrationNumber.Length != 0)
				{
					//if it is supposed to have alphas and does not
					//then say it is invalid and why
					if(this._localSupplier.LabelUnitIDTrimIndicator)	
					{
						bool bAlpha = false;
						nCount = unitID.Length;
						for(int nCtr = 0; nCtr<nCount; nCtr++)
						{
							if(char.IsLetter(unitID, nCtr))
							{
								bAlpha = true;
								break;
							}
							
						}
						if(!bAlpha)
						{
							validMessage = "Unit ID must contain alpha characters";
							return;
						}
					}
						//Unit ID should not contain alpha characters
					else
					{
						bool bAlpha = false;
						//BR_1.05 If the unit is codabar and the id is thirteen
						//characters long then the last character can be 
						//an alpha regardless of the local supplier setting,
						//this would indicate a split unit.
						if(this.LabelTypeCode == "C" && unitID.Length == 13)
						{
							int unitIDLenLessLastChar = unitID.Length - 1;
							for(int nCtr = 0; nCtr<unitIDLenLessLastChar; nCtr++)
							{
								if(nCtr <6)
								{
									if(char.IsLetter(unitID, nCtr))
									{
										bAlpha = true;
										break;
									}
								}
							}
						}
						else
						{
							for(int nCtr = 0; nCtr<unitID.Length; nCtr++)
							{
								if(nCtr<6)
								{
									if(char.IsLetter(unitID, nCtr))
									{
										bAlpha = true;
										break;
									}
								}
							}
						}
						if(bAlpha)
						{
							validMessage = "Unit ID must contain only numbers.";
							return;
						}
						
					}
					//Check if they should have a prefix
					if(this._localSupplier.FacilityPrefix.Length ==2)
					{
						if(unitID.Length >= 2)
						{
							if(unitID.Substring(0,2) != this._localSupplier.FacilityPrefix)
							{
								validMessage = "Unit ID does not contain the correct facility prefix.";
							}
						}
					}
				}
				else
				{
					validMessage = "There is no FDA Registration number.";
				}
			}
			else
			{
				validMessage = "There is no FDA Registration number.";
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/23/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1476"> 
		///		<ExpectedInput>BOL.LocalSupplier</ExpectedInput>
		///		<ExpectedOutput>BOL.LocalSupplier</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1477"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Local Supplier
		/// </summary>
		public BOL.LocalSupplier LocalSupplier
		{
			get 
			{
				if (this._localSupplier == null) 
				{
					if (this.UnitFdaRegistrationNumber.Trim() != String.Empty)
					{
						_localSupplier = new BOL.LocalSupplier(this.UnitFdaRegistrationNumber,true);
					}
					else
					{
						return null;
					}
				}
				return _localSupplier; 
			}
			set 
			{ 
				//TODO: No broken rule exists in table
				RuleBroken(Method.GetCurrentMethod().Name,(value == null));
				_localSupplier = value;
			}
		}
        
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5120"> 
		///		<ExpectedInput>decimal</ExpectedInput>
		///		<ExpectedOutput>decimal</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5121"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OriginalVolume
		/// </summary>
		public decimal OriginalVolume 
		{
			get
			{
				if (_bloodUnitDataRow.IsNull(TABLE.BloodUnit.OriginalVolume))
				{
					return 0.0m;
				}
				return (decimal)_bloodUnitDataRow[TABLE.BloodUnit.OriginalVolume];
			}
			set
			{
				_bloodUnitDataRow[TABLE.BloodUnit.OriginalVolume] = value;
				this.OnVolumeChanged();		//Added 3/1/2004 for ModifyUnits - Lohse
			} 

		}
        
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5122"> 
		///		<ExpectedInput>decimal</ExpectedInput>
		///		<ExpectedOutput>decimal</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5123"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// DiscardedPlasmaVolume
		/// </summary>
		public decimal DiscardedPlasmaVolume 
		{
			get
			{
				if (_bloodUnitDataRow.IsNull(TABLE.BloodUnit.DiscardedPlasmaVolume))
				{
					return 0.0m;
				}
				return (decimal)_bloodUnitDataRow[TABLE.BloodUnit.DiscardedPlasmaVolume];
			}
			set
			{
				_bloodUnitDataRow[TABLE.BloodUnit.DiscardedPlasmaVolume] = value;
				this.OnDiscardedPlasmaVolumeChanged();
			} 

		}
        
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5124"> 
		///		<ExpectedInput>short</ExpectedInput>
		///		<ExpectedOutput>short</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5125"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PooledUnitsCount
		/// </summary>
		public short PooledUnitsCount 
		{
			get
			{
				DataTable dtUnit = DAL.BloodUnit.GetBloodUnitByGuid(this.BloodUnitGuid);

				return((short)dtUnit.Rows[0][Common.DatabaseConstants.ArtificialColumnNames.PooledUnitsCount]);
			}
		}
        
		/// <summary>
		/// Returns true if the BloodUnit exists on an Unconfirmed Outgoing Invoice
		/// </summary>
		public bool PendingTransferIndicator
		{
			get
			{
				if (this._bloodUnitDataRow.Table.Columns.Contains(ARTIFICIAL.PendingTransferIndicator))
				{
					return(System.Convert.ToBoolean(this._bloodUnitDataRow[ARTIFICIAL.PendingTransferIndicator]));
				}
				else
				{
					//Query the database again, since this condition means a BU constructor is not returning our field
					System.Data.DataTable dtUnit = DAL.BloodUnit.GetBloodUnitByGuid(this.BloodUnitGuid);
					if (dtUnit.Rows.Count == 0)		//If the Unit wasn't found, it's probably new, so false is valid
					{
						return(false);
					}
					else
					{
						return(System.Convert.ToBoolean(dtUnit.Rows[0][ARTIFICIAL.PendingTransferIndicator]));
					}
				}
			}
		}


		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5128"> 
		///		<ExpectedInput>DateTime</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5129"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// UnitInDate
		/// </summary>
		public System.DateTime UnitInDate 
		{
			get
			{
				if (_bloodUnitDataRow.IsNull(TABLE.BloodUnit.UnitInDate))
				{
					return System.DateTime.MinValue;
				}
				return (System.DateTime)_bloodUnitDataRow[TABLE.BloodUnit.UnitInDate];
			}
			set
			{
				_bloodUnitDataRow[TABLE.BloodUnit.UnitInDate] = value;
			} 

		}
        
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5130"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5131"> 
		///		<ExpectedInput>invalid string</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// UnitContainerLotNumber
		/// </summary>
		public string UnitContainerLotNumber 
		{
			get
			{
				if (_bloodUnitDataRow.IsNull(TABLE.BloodUnit.UnitContainerLotNumber))
				{
					return String.Empty;
				}
				return _bloodUnitDataRow[TABLE.BloodUnit.UnitContainerLotNumber].ToString();
			}
			set
			{
				if (value == null || value.Length > 15)
				{
					throw new ArgumentException("Invalid UnitContainerLotNumber argument", "value");
				}
				_bloodUnitDataRow[TABLE.BloodUnit.UnitContainerLotNumber] = value;
			} 

		}
         	
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5134"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5135"> 
		///		<ExpectedInput>invalid string</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LabelTypeCode
		/// </summary>
		public string LabelTypeCode 
		{
			get
			{
				if (_bloodUnitDataRow.IsNull(TABLE.BloodUnit.LabelTypeCode))
				{
					return String.Empty;
				}
				return _bloodUnitDataRow[TABLE.BloodUnit.LabelTypeCode].ToString();
			}
			set
			{
				if (value == null || value.Length > 1)
				{
					throw new ArgumentException("Invalid LabelTypeCode argument", "value");
				}
				_bloodUnitDataRow[TABLE.BloodUnit.LabelTypeCode] = value;
			} 

		}
        
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5136"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5137"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LabelUnitId
		/// </summary>
		public string LabelUnitId 
		{
			get
			{
				if (_bloodUnitDataRow.IsNull(TABLE.BloodUnit.LabelUnitId))
				{
					return String.Empty;
				}
				return _bloodUnitDataRow[TABLE.BloodUnit.LabelUnitId].ToString();
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7379"> 
		///		<ExpectedInput>Valid Id, false</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7380"> 
		///		<ExpectedInput>Invlaid Id, false</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Allows the LabelUnitId to be set without having to use a regular expression for validation
		/// </summary>
		/// <param name="strLabelUnitId"></param>
		/// <param name="useRegEx"></param>
		public void SetLabelUnitId(string strLabelUnitId, bool useRegEx)
		{
			if (!useRegEx)
			{
				RuleBroken(Method.GetCurrentMethod().Name, false);
				_bloodUnitDataRow[TABLE.BloodUnit.LabelUnitId] = strLabelUnitId;
				return;
			}

			if(this.BarcodeType == Common.BarcodeType.Codabar)
			{
				if (!this._codabarScannedUnitID.IsMatch(strLabelUnitId))
				{
					this.RuleBroken(Method.GetCurrentMethod().Name, true);
					throw new BOL.BusinessObjectException("Invalid codabar unit id");
				}
			}
			else if(this.BarcodeType == Common.BarcodeType.ISBT128)
			{
				if(strLabelUnitId.ToString().IndexOf("=") > -1)
				{
					if(!this._scannedISBTUnitID.IsMatch(strLabelUnitId))
					{
						this.RuleBroken(Method.GetCurrentMethod().Name, true);
						throw new BOL.BusinessObjectException("Invalid ISBT 128 unit id");
					}
				}
				else
				{
					if(!this._isbtUnitID.IsMatch(strLabelUnitId))
					{
						this.RuleBroken(Method.GetCurrentMethod().Name, true);
						throw new BOL.BusinessObjectException("Invalid ISBT 128 unit id");
					}
				}
			}
			else 
			{
				this.RuleBroken(Method.GetCurrentMethod().Name, true);
				throw new BOL.BusinessObjectException("Must set the barcode type before setting the label unit id");
			}
                
			if(this.ProductCode.Length > 0)
			{
				this.BloodUnitFinance.LoadBloodUnitFinancialData(this.LabelUnitId, this.ProductCode);
			}
				
			RuleBroken(Method.GetCurrentMethod().Name, strLabelUnitId.ToString() == String.Empty);
			_bloodUnitDataRow[TABLE.BloodUnit.LabelUnitId] = strLabelUnitId;
		}
        
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5138"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5139"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// EyeReadableUnitId
		/// </summary>
		public string EyeReadableUnitId 
		{
			get
			{
				if (_bloodUnitDataRow.IsNull(TABLE.BloodUnit.EyeReadableUnitId))
				{
					return String.Empty;
				}
				return _bloodUnitDataRow[TABLE.BloodUnit.EyeReadableUnitId].ToString();
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7381"> 
		///		<ExpectedInput>Invalid EyereableUnitId, false</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7382"> 
		///		<ExpectedInput>Invalid EyereableUnitId, false</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Allows the EyeReadableUnitId to be set without having to use a regular expression for validation
		/// </summary>
		/// <param name="strEyeReadable"></param>
		/// <param name="useRegEx"></param>
		public void SetEyeReadableUnitId(string strEyeReadable, bool useRegEx)
		{
			if(strEyeReadable.Trim().Length > 13)
			{
				this.RuleBroken(Method.GetCurrentMethod().Name, true);
				throw new BOL.BusinessObjectException("Invalid unit id");
			}

			if (!useRegEx) 
			{
				RuleBroken(Method.GetCurrentMethod().Name, false);
				_bloodUnitDataRow[TABLE.BloodUnit.EyeReadableUnitId] = strEyeReadable;
				return;
			}
			//
			if(this.BarcodeType == Common.BarcodeType.Codabar)
			{
				if (!this._codabarEyeReadableUnitID.IsMatch(strEyeReadable))
				{
					this.RuleBroken(Method.GetCurrentMethod().Name, true);
					throw new BOL.BusinessObjectException("Invalid codabar unit id");
				}
				if(strEyeReadable.Trim().Length == 6)
				{
					bool bAlpha = false;
						
					if(char.IsLetter(strEyeReadable.Substring(0,1), 0))
					{
						bAlpha = true;
					}
					if(!bAlpha)
					{
						this.RuleBroken(Method.GetCurrentMethod().Name, true);
						throw new BOL.BusinessObjectException("Invalid codabar unit id");							
					}
				}
			}
			else if(this.BarcodeType == Common.BarcodeType.ISBT128)
			{
				if(strEyeReadable.ToString().IndexOf("=") > -1)
				{
					if(!this._scannedISBTUnitID.IsMatch(strEyeReadable))
					{
						this.RuleBroken(Method.GetCurrentMethod().Name, true);
						throw new BOL.BusinessObjectException("Invalid ISBT 128 unit id");
					}
				}
				else
				{
					if(!this._isbtUnitID.IsMatch(strEyeReadable))
					{
						this.RuleBroken(Method.GetCurrentMethod().Name, true);
						throw new BOL.BusinessObjectException("Invalid ISBT 128 unit id");
					}
				}
			}
			else 
			{
				this.RuleBroken(Method.GetCurrentMethod().Name, true);
				throw new BOL.BusinessObjectException("Must set the barcode type before setting the label unit id");
			}
			_bloodUnitDataRow[TABLE.BloodUnit.EyeReadableUnitId] = strEyeReadable;
			if(this.ProductCode.Length > 0 && strEyeReadable.Length > 0)
			{
				this.BloodUnitFinance.LoadBloodUnitFinancialData(this.EyeReadablePrefix, this.ProductCode);
			}
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5209"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5210"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ProcessingTechId
		/// </summary>
		public string ProcessingTechId
		{
			get
			{
				return _bloodUnitDataRow[TABLE.BloodUnit.ProcessingTechId].ToString();
			}
			set
			{
				_bloodUnitDataRow[TABLE.BloodUnit.ProcessingTechId] = value;
			}
		}	

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/23/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1481"> 
		///		<ExpectedInput>Numeric string of 7 characters</ExpectedInput>
		///		<ExpectedOutput>Valid unit FDA registration number</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1482"> 
		///		<ExpectedInput>Alphabetic string of 7 characters</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Unit FDA Registration Number
		/// </summary>
		public string UnitFdaRegistrationNumber
		{
			get 
			{
				if (_bloodUnitDataRow.IsNull(TABLE.BloodUnit.UnitFdaRegistrationNumber))
				{
					return String.Empty;
				}
				return (System.String)_bloodUnitDataRow[TABLE.BloodUnit.UnitFdaRegistrationNumber];
			}
			set 
			{ 
				Regex _regexFDAReg = Common.RegularExpressions.FDARegistrationNumber();
			
				//
				if ((value.Trim().Length > 0) && !_regexFDAReg.IsMatch(value.Trim()))
				{
					string errorMessage = Common.StrRes.SysErrMsg.Common.InvalidPropertyOrMethod(
						Common.Utility.ParseStringwithSpaces(Method.GetCurrentMethod().Name, true),
						value).ResString;
					throw new BOL.BusinessObjectException(errorMessage);
				}
                
				RuleBroken(Method.GetCurrentMethod().Name,((this.BarcodeType==Common.BarcodeType.Codabar) && (value.Trim().Length==0)));	
				_bloodUnitDataRow[TABLE.BloodUnit.UnitFdaRegistrationNumber] = value.Trim();
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/23/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1504"> 
		///		<ExpectedInput>Valid BarcodeType</ExpectedInput>
		///		<ExpectedOutput>Valid BarcodeType</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1505"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Barcode type BR_1.04
		/// </summary>
		public new Common.BarcodeType BarcodeType
		{
			get 
			{
				if(_bloodUnitDataRow == null)
				{
					return this._barcodeType;
				}
				else
				{
					return Common.Utility.GetBarcodeTypeFromString(_bloodUnitDataRow[TABLE.BloodUnit.LabelTypeCode].ToString());
				}
			}
			set 
			{ 
				if(_bloodUnitDataRow == null)
				{
					this._barcodeType = value;
				}
				else
				{
					_bloodUnitDataRow[TABLE.BloodUnit.LabelTypeCode] = Common.Utility.GetBarcodeTypeCodeCharFromEnum(value);
				}
			}
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5140"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5141"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// IncomingShipmentGuid
		/// </summary>
		public System.Guid IncomingShipmentGuid
		{
			get
			{
				if (_bloodUnitDataRow.IsNull(TABLE.BloodUnitShipment.IncomingShipmentGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)_bloodUnitDataRow[TABLE.BloodUnitShipment.IncomingShipmentGuid];
			}
			set
			{
				_bloodUnitDataRow[TABLE.BloodUnitShipment.IncomingShipmentGuid] = value;
			} 

		}
        
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5142"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5143"> 
		///		<ExpectedInput>LocalSupplier not set</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// EyeReadablePrefix
		/// </summary>
		public string EyeReadablePrefix 
		{
			get
			{
				if(this._localSupplier != null)
				{
					return this._localSupplier.FacilityPrefix;
				}
				else
				{
					return string.Empty;
				}
			}
			set
			{
				if (value == null || value.Length > 2)
				{
					throw new ArgumentException("Invalid EyeReadablePrefix argument", "value");
				}
				this._localSupplier.FacilityPrefix = value;
			} 
		}
      
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5144"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5145"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// EntryMethodCode
		/// </summary>
		public string EntryMethodCode 
		{
			get
			{
				if (_bloodUnitDataRow.IsNull(TABLE.BloodUnit.EntryMethodCode))
				{
					return String.Empty;
				}
				return _bloodUnitDataRow[TABLE.BloodUnit.EntryMethodCode].ToString();
			}
			set
			{
				if (value == null || value.Length > 1)
				{
					throw new ArgumentException("Invalid EntryMethodCode argument", "value");
				}
				_bloodUnitDataRow[TABLE.BloodUnit.EntryMethodCode] = value;
			} 
		}
	
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/23/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1489"> 
		///		<ExpectedInput>Patient</ExpectedInput>
		///		<ExpectedOutput>Patient</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1490"> 
		///		<ExpectedInput>Null</ExpectedInput>
		///		<ExpectedOutput>broken rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  Directed Patient
		/// </summary>
		public Patient DirectedPatient
		{
			get
			{
				if (this._directedPatient == null) 
				{
					_directedPatient = new BOL.Patient();
				}
				return this._directedPatient;
			}
			set
			{
				RuleBroken(Method.GetCurrentMethod().Name,(value == null));
				this._directedPatient = value;
				if(this.BloodUnitMedia != null && this._directedPatient != null)
				{
					this.BloodUnitMedia.DirectedPatientGuid = this._directedPatient.PatientGuid;
				}
				this.IsDirty = true;
			}
		}
        
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5146"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5147"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PatientAssignments
		/// </summary>
		public System.Data.DataTable PatientAssignments
		{
			get
			{
				if(this.BloodUnitGuid == System.Guid.Empty)
				{
					throw new BusinessObjectException("You must set the blood unit GUID");
				}
				this._patientAssignments = BOL.PatientAssignments.GetPatientAssignments(this.BloodUnitGuid);
				return this._patientAssignments;
			}
		}
	
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5148"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5149"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CMVNegative
		/// </summary>
		public bool CmvNegative
		{
			get
			{
				string val = BOL.BloodUnitTest.GetBloodUnitTestResultValue(this.BloodUnitGuid, System.Convert.ToInt32(Common.TestType.CMV), this.DivisionCode);
				if(val.Trim().Length == 0)
				{
					return false;
				}
				else if (val.Trim() == "N")
				{
					return true;
				}
				else
				{
					return false;
				}
			}
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5150"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5151"> 
		///		<ExpectedInput>invalid string</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///		/// <summary>
		/// Division code
		/// </summary>
		public string DivisionCode 
		{
			get
			{
				if (this._bloodUnitDataRow.IsNull(TABLE.BloodUnit.DivisionCode))
				{
					return Common.LogonUser.LogonUserDivisionCode;
				}
				return (string)this._bloodUnitDataRow[TABLE.BloodUnit.DivisionCode];
			}
			set
			{
				if (value == null || value.Length > 5)
				{
					throw new ArgumentException("Invalid DivisionCode argument", "value");
				}
				this._bloodUnitDataRow[TABLE.BloodUnit.DivisionCode] = value;
			} 
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="5152"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5153"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SickleCellNegative
		/// </summary>
		public bool SickleCellNegative
		{
			get
			{
				string val = BOL.BloodUnitTest.GetBloodUnitTestResultValue(this.BloodUnitGuid, System.Convert.ToInt32(Common.TestType.SickleCell), this.DivisionCode);
				if(val.Trim().Length == 0)
				{
					return false;
				}
				else if (val.Trim() == "N")
				{
					return true;
				}
				else
				{
					return false;
				}
			}
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5154"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5155"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Used in UC 1 to show all RBC antigens
		/// </summary>
		public string RbcAntigens
		{
			get
			{
				return BOL.BloodUnit.GetRBCAntigens(this.BloodUnitGuid);
			}
		}
		/// <summary>
		///     Used internally to return the blood unit information without any instance information
		/// </summary>
		/// <param name="bloodUnitGuid" type="System.Guid">
		///     <para>
		///         
		///     </para>
		/// </param>
		/// <returns>
		///     A string value...
		/// </returns>
		private static string GetRBCAntigens(System.Guid bloodUnitGuid)
		{
			ArrayList rbcAntigens = new ArrayList();
			string antigenName = string.Empty;
			System.Data.DataTable dtspecialTests = BOL.BloodUnitTest.GetBloodUnitSpecialTest(bloodUnitGuid);
			if(dtspecialTests.Rows.Count > 0)
			{
				return dtspecialTests.Rows[0][0].ToString();
			}	
			else
			{
				return string.Empty;
			}
		}
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5156"> 
		///		<ExpectedInput>Common.RecordStatusCode</ExpectedInput>
		///		<ExpectedOutput>Common.RecordStatusCode</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5157"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The status of the record
		/// </summary>
		public Common.RecordStatusCode RecordStatusCode
		{
			get
			{
				return _recordStatusCode;
			}
			set
			{
				_recordStatusCode = value;
			}
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5158"> 
		///		<ExpectedInput>BloodUnitStatus</ExpectedInput>
		///		<ExpectedOutput>BloodUnitStatus</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5159"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodUnitStatus
		/// </summary>
		public virtual BloodUnitStatus BloodUnitStatus
		{
			get
			{
				if (this._bloodUnitStatus == null) 
				{
					_bloodUnitStatus = new BloodUnitStatus(this.BloodUnitGuid);
				}
				return this._bloodUnitStatus;
			}
			set
			{
				// TODO: Implement this member.
				this._bloodUnitStatus = value;
			}
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5160"> 
		///		<ExpectedInput>BloodUnitFinance</ExpectedInput>
		///		<ExpectedOutput>BloodUnitFinance</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5161"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodUnitFinance
		/// </summary>
		public virtual BloodUnitFinance BloodUnitFinance
		{
			get
			{
				if (this._bloodUnitFinance == null) 
				{
					_bloodUnitFinance =  new BloodUnitFinance(this.BloodUnitGuid);
				}
				return this._bloodUnitFinance;
			}
			set
			{
				// TODO: Implement this member.
				this._bloodUnitFinance = value;
			}
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5162"> 
		///		<ExpectedInput>BloodUnitMedia</ExpectedInput>
		///		<ExpectedOutput>BloodUnitMedia</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5163"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodUnitMedia
		/// </summary>
		public virtual BOL.BloodUnitMedia BloodUnitMedia
		{
			get
			{
				if (this._bloodUnitMedia == null) 
				{
					_bloodUnitMedia = new BloodUnitMedia(this.BloodUnitGuid);
				}
				return this._bloodUnitMedia;
			}
			set
			{
				// TODO: Implement this member.
				this._bloodUnitMedia = value;
			}
		}
        
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/23/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1499"> 
		///		<ExpectedInput>Specimen[]</ExpectedInput>
		///		<ExpectedOutput>Specimen[]</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1500"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Patient specimen object associated with the blood unit
		/// </summary>
		public BOL.Specimen[] PatientSpecimens
		{
			get
			{
				return this._patientSpecimens;
			}
			set
			{
				this._patientSpecimens = value;
			}
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5166"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>BOL.Patient[]</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5167"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// AssignedToPatients
		/// </summary>
		public System.Collections.ArrayList AssignedToPatients
		{
			get 
			{
				if (_assignedToPatients == null) 
				{
					GetAssignedToPatients();
				}
				return _assignedToPatients;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/11/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5168"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5169"> 
		///		<ExpectedInput>empty string</ExpectedInput>
		///		<ExpectedOutput>broken rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// The blood unit's check character 
		/// </summary>
		/// 
		public  string CheckCharacter
		{
			get
			{
				return this._checkCharacter;
			}
			set
			{
				RuleBroken("CheckCharacter", value == String.Empty); 
				this._checkCharacter = value;
			}
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/30/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4055"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4350"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Determines whether a unit can be freed from its restriction.
		/// If any of the conditions below are met, the unit cannot be 
		/// released and the reason(s) for ineligibility is/are returned.
		/// Implements BR_71.01, BR_71.02 and BR_71.03
		/// </summary>
		/// <returns>string indicating what, if any, conditions were not met</returns>
		public string IsEligibleForCrossover() 
		{
			StringBuilder message = new StringBuilder();

			// ***************************************************************************************************************
			// If any of the following conditions are met, the unit cannot be released:
			// ***************************************************************************************************************
			// BR_71.01: check if the unit is autologous
			if(IsAutologous(DonationType.DonationTypeCode))
			{
				message.Append("\n");
				message.Append(StrRes.InfoMsg.UC071.IsAutologous().ResString);
			}
				// If not autologous, make sure the unit is directed
			else if(!IsDirected(DonationType.DonationTypeCode))
			{
				message.Append("\n");
				message.Append(StrRes.InfoMsg.UC071.NotDirected().ResString);
			}
			// ***************************************************************************************************************
			// BR_71.01: check the donation type, it must be Volunteer Directed Eligible For Crossover
			Common.DonationType donationType = Common.Utility.GetDonationTypeCodeEnumFromChar(DonationType.DonationTypeCode);
			if(donationType != Common.DonationType.VolunteerDirectedEligibeForCrossover)
			{
				message.Append("\n");
				message.Append(StrRes.InfoMsg.UC071.InvalidDonationType().ResString);
			}
			// ***************************************************************************************************************
			// BR_71.02: check if the unit is in final status
			if(BloodUnitStatus.IsFinal())
			{
				message.Append("\n");
				message.Append(StrRes.InfoMsg.UC071.InFinalStatus().ResString);
			}
			// ***************************************************************************************************************
			// BR_71.03: check if the unit is assigned
			if(DAL.BloodUnit.GetAssignedPatientCount(BloodUnitGuid) > 0)
			{
				message.Append("\n");
				message.Append(StrRes.InfoMsg.UC071.IsAssigned().ResString);
			}
			// ***************************************************************************************************************

			return message.ToString();
		}

		#endregion

		#region Events

		/// <summary>
		/// This event and delegate were added on 3/1/2004 for ModifyUnits
		/// </summary>
		public event EventHandler VolumeChanged;

		/// <summary>
		/// Fires the VolumeChanged event when OrignalVolume is modified
		/// </summary>
		protected void OnVolumeChanged()
		{
			if (VolumeChanged != null)
				this.VolumeChanged(this, System.EventArgs.Empty);
		}

		/// <summary>
		/// This event and delegate were added on 3/24/2004 for ModifyUnits
		/// </summary>
		public event EventHandler DiscardedPlasmaVolumeChanged;

		/// <summary>
		/// Fires the DiscardedPlasmaVolumeChanged event when DiscardedPlasmaVolume is modified
		/// </summary>
		protected void OnDiscardedPlasmaVolumeChanged()
		{
			if (DiscardedPlasmaVolumeChanged != null)
				this.DiscardedPlasmaVolumeChanged(this, System.EventArgs.Empty);
		}
		/// <summary>
		/// OnColumnChanged
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="eventArgs"></param>
		private void OnColumnChanged(object sender, DataColumnChangeEventArgs eventArgs)
		{
			this.IsDirty = true;
		}
        

		/// <summary>
		/// BiohazardousWarningChanged
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected void BiohazardousWarningChanged(object sender, System.EventArgs e)
		{
			if (
				(this.BloodUnitMedia.TestingStatusId == (int) Common.TestingStatus.NotFullyTested) || 
				(this.BloodUnitMedia.TestingStatusId == (int) Common.TestingStatus.Positive) || 
				(this.DonationType.DonationTypeCode == Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.ForDirectedDonorUseOnlyBiohazardous)) ||
				(this.DonationType.DonationTypeCode == Common.Utility.GetDonationTypeCodeCharFromEnum(Common.DonationType.ForAutologousUseOnlyBiohazardous))
				)
				this.BloodUnitStatus.BiohazardousWarningIndicator = true;
			else
				this.BloodUnitStatus.BiohazardousWarningIndicator = false;
		}


		#endregion

		#region Static Methods

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5170"> 
		///		<ExpectedInput>donationtypecode</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5171"> 
		///		<ExpectedInput>invalid donationtypecode</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// IsAutologous
		/// </summary>
		/// <param name="donationTypeCode"></param>
		/// <returns></returns>
		public static bool IsAutologous(char donationTypeCode) 
		{
			Common.DonationType donationType = Common.Utility.GetDonationTypeCodeEnumFromChar(donationTypeCode);

			return (donationType == Common.DonationType.AutologousEligibleForCrossover
				|| donationType == Common.DonationType.ForAutologousUseOnly
				|| donationType == Common.DonationType.ForAutologousUseOnlyBiohazardous);
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5172"> 
		///		<ExpectedInput>donationTypeCode</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5173"> 
		///		<ExpectedInput>invalid donationTypeCode</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// IsAllogenic
		/// </summary>
		/// <param name="donationTypeCode"></param>
		/// <returns></returns>
		public static bool IsAllogenic(char donationTypeCode)
		{
			Common.DonationType donationType = Common.Utility.GetDonationTypeCodeEnumFromChar(donationTypeCode);

			return (donationType == Common.DonationType.VolunteerAllogeneicDonor);
		}


		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5174"> 
		///		<ExpectedInput>donationTypeCode</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5175"> 
		///		<ExpectedInput>invalid donationTypeCode</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// IsDirected
		/// </summary>
		/// <param name="donationTypeCode"></param>
		/// <returns></returns>
		public static bool IsDirected(char donationTypeCode) 
		{
			Common.DonationType donationType = Common.Utility.GetDonationTypeCodeEnumFromChar(donationTypeCode);

			return (donationType == Common.DonationType.VolunteerDirectedEligibeForCrossover
				|| donationType == Common.DonationType.PaidDirectedEligibleForCrossover
				|| donationType == Common.DonationType.ForDirectedDonorUseOnly
				|| donationType == Common.DonationType.ForDirectedDonorUseOnlyLimitedExposure
				|| donationType == Common.DonationType.ForDirectedDonorUseOnlyMedicalException
				|| donationType == Common.DonationType.ForDirectedDonorUseOnlyBiohazardous
				|| donationType == Common.DonationType.DedicatedDonor
				|| donationType == Common.DonationType.DesignatedDonor
				|| donationType == Common.DonationType.VolunteerResearchDonor
				|| donationType == Common.DonationType.VolunteerSourceDonor
				|| donationType == Common.DonationType.VolunteerTherapeuticCollection
				|| donationType == Common.DonationType.PaidAllogeneicCollection
				|| donationType == Common.DonationType.PaidResearchCollection
				|| donationType == Common.DonationType.PaidSourceCollection);
		}
        		
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5176"> 
		///		<ExpectedInput>unit id, productcode, division, divisionid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5177"> 
		///		<ExpectedInput>invalid parameters</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetBloodUnitGuid - DataAccessLayerExceptino if not found
		/// </summary>
		/// <param name="labelUnitID"></param>
		/// <param name="unitProductCode"></param>
		/// <param name="divisionCode"></param>
		/// <param name="unitDivisionId">Unit Division ID - needed for alloQUAT finding ISBT only</param>
		/// <returns></returns>
		public static System.Guid GetBloodUnitGuid(string labelUnitID, string unitProductCode, string divisionCode, string unitDivisionId)
		{
			return DAL.BloodUnit.GetBloodUnitGuid(labelUnitID,unitProductCode,divisionCode, unitDivisionId);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/08/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="5786"> 
		///		<ExpectedInput>unitid, productcode, division, divisionid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5787"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetBloodUnitGuid
		/// </summary>
		/// <param name="labelUnitID"></param>
		/// <param name="unitProductCode"></param>
		/// <param name="divisionCode"></param>
		/// <param name="unitDivisionId">Unit Division ID - needed for alloQUAT finding ISBT only</param>
		/// <param name="donationTypeCode">Donation Type COde - needed for ISBT only</param>
		/// <param name="searchCriteria"></param>
		/// <returns></returns>
		public static DataTable GetBloodUnitsByIdAndProduct
			(
			string labelUnitID, 
			string unitProductCode, 
			string divisionCode, 
			string unitDivisionId, 
			string donationTypeCode,
			Common.AdditionalBloodUnitSearchCriteria searchCriteria
			)
		{
            return DAL.BloodUnit.GetBloodUnitsByIdAndProduct
                    (
                    labelUnitID, unitProductCode, divisionCode, unitDivisionId, donationTypeCode, searchCriteria
                    );

		}

		///<Developers>
		///<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/21/2002</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="685"> 
		///		<ExpectedInput>Valid eye readable unit ID, product code</ExpectedInput>
		///		<ExpectedOutput>Valid data table</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="1725"> 
		///		<ExpectedInput>empty strings for eye readable unit ID, product code</ExpectedInput>
		///		<ExpectedOutput>Empty data table</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Loads all the blood unit details for a worklist
		/// </summary>
		/// <param name="eyeReadableUnitID"></param>
		/// <param name="bloodProductGuid"></param>
		/// <returns></returns>
		public static DataTable GetBloodUnitDetails(string eyeReadableUnitID, System.Guid bloodProductGuid)
		{
			//TODO:receive 2 data tables. 1-with blood unit details. 2-with assignedto patient GUIDs
			//Set the PatientAssignments property of this class via the GUID array from the 2nd datatable

			return DAL.BloodUnit.GetBloodUnitDetails(eyeReadableUnitID, bloodProductGuid);
		}

		
		///<Developers>
		///	<Developer>Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/21/2005</CreationDate>
		/// <summary>
		/// Checks if the blood unit exists from a given eyereadableunit,
		/// product code, and division code.
		/// </summary>
		/// 
		///<Case type="0" testid ="8355"> 
		///		<ExpectedInput>unitid, productguid, productcode, bloodunitguid</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8356"> 
		///		<ExpectedInput>In Valid unitid, productguid, productcode, bloodunitguid</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///
		/// <param name="eyeReadableUnitId" type="string">
		///     <para>
		///         
		///     </para>
		/// </param>
		/// <param name="bloodProductGuid" type="System.Guid"></param>
		/// <param name="productCode" type="string"></param>
		/// <param name="bloodUnitGuid" type="ref System.Guid"></param>
		/// <returns>
		///     A bool value...
		/// </returns>
		public static bool BloodUnitExists(string eyeReadableUnitId, System.Guid bloodProductGuid, string productCode, ref System.Guid bloodUnitGuid)
		{
			//TODO:receive 2 data tables. 1-with blood unit details. 2-with assignedto patient GUIDs
			//Set the PatientAssignments property of this class via the GUID array from the 2nd datatable
			
			System.Data.DataTable dt = gov.va.med.vbecs.DAL.BloodUnit.GetBloodUnitDetails(eyeReadableUnitId, bloodProductGuid, productCode);
			if(dt.Rows.Count > 0)
			{
				bloodUnitGuid = (System.Guid) dt.Rows[0][TABLE.BloodUnit.BloodUnitGuid];
				return true;
			}
			else
			{
				return false;
			}
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5182"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5183"> 
		///		<ExpectedInput>invalid Guid</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks if a blood unit already exists for a given Blood Unit Guid
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static bool BloodUnitExists(System.Guid bloodUnitGuid)
		{
			// CR 3006
			return DAL.BloodUnit.BloodUnitExists( bloodUnitGuid );
		}

		///<Developers>
		///	<Developer>Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>03/19/2007</CreationDate>
		///
		///<Case type="0" testid ="8477"> 
		///		<ExpectedInput>unitid, productguid, productcode, bloodunitguid</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8478"> 
		///		<ExpectedInput>In Valid unitid, productguid, productcode, bloodunitguid</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///
		/// <summary>
		/// Checks if the unit ever existed form an incoming shipment and has since been modified so it is not in current inventory
		/// </summary>
		/// <param name="eyeReadableUnitId"></param>
		/// <param name="bloodProductGuid"></param>
		/// <param name="productCode"></param>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static bool IncomingShipmentBloodUnitExists(string eyeReadableUnitId, System.Guid bloodProductGuid, string productCode, ref System.Guid bloodUnitGuid)
		{
			System.Data.DataTable dt = gov.va.med.vbecs.DAL.BloodUnit.GetIncomingShipmentBloodUnitDetails(eyeReadableUnitId, bloodProductGuid, productCode);
			if(dt.Rows.Count > 0)
			{
				if(dt.Columns.Count == 1)
				{
					return false;
				}
				else
				{
					bloodUnitGuid = (System.Guid) dt.Rows[0][TABLE.BloodUnit.BloodUnitGuid];
					return true;
				}
			}
			else
			{
				return false;
			}
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5180"> 
		///		<ExpectedInput>productCode</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5181"> 
		///		<ExpectedInput>string less than 8 chars</ExpectedInput>
		///		<ExpectedOutput>Empty string</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the UnitDivisionId for the incoming productcode
		/// </summary>
		/// <param name="productCode"></param>
		/// <returns></returns>
		public static string GetUnitDivisionId(string productCode)
		{
			if (productCode.Length < 8)
			{
				return string.Empty;
			}
			else
			{
				return productCode.Trim().Substring(6,2);
			}
		}

		/// <summary>
		/// Returns the DonationTypeCode for the incoming productcode
		/// </summary>
		/// <param name="productCode"></param>
		/// <returns></returns>
		public static string GetDonationTypeCode(string productCode)
		{
			if (productCode.Length < 6)
			{
				return string.Empty;
			}
			else
			{
				return productCode.Trim().Substring(5,1);
			}
		}

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/24/2002</CreationDate>
		/// <summary>
		/// Returns a DataTable of Blood Unit information
		/// </summary>
        /// <param name="bloodUnitGuid">BloodUnitGuid to check - can return zero rows if BRs in sproc fail</param>
		/// <returns></returns>
		public static DataTable GetLimitedBloodUnit(System.Guid bloodUnitGuid)
		{
            //Defect 251426 - removed ShipmentGuid - so this will primarily be used for re-confirmation, allowing the UI to offer better error msgs
			return DAL.BloodUnit.GetLimitedBloodUnit(bloodUnitGuid);
		}


        /// <summary>
        /// Gets a list of Limited Blood units for a specified Shipment/Invoice. 
        /// Created for Defect 251426
        /// </summary>
        /// <param name="shipmentGuid">IncomingShipmientGuid</param>
        /// <param name="allowNegativeRh">Pull back Negative RH units in the list (if AntiD is setup)</param>
        /// <returns></returns>
	    public static DataTable GetLimitedBloodUnitsByShipment(System.Guid shipmentGuid, bool allowNegativeRh)
        {
            return DAL.BloodUnit.GetLimitedBloodUnitsByDivision(Common.LogonUser.LogonUserDivisionCode, allowNegativeRh, shipmentGuid);

        }

	    ///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		/// <summary>
		/// GetLimitedBloodUnitsByDivision
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="allowNegativeRh">If user turns off AntiD testing in UC2, they shouldn't be allowed to select negatvie units - else violate BR2.07</param>
		/// <returns></returns>
		public static DataTable GetLimitedBloodUnitsByDivision(string divisionCode, bool allowNegativeRh)
		{
			return DAL.BloodUnit.GetLimitedBloodUnitsByDivision(divisionCode, allowNegativeRh, Guid.Empty);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/23/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5186"> 
		///		<ExpectedInput>division, bool</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5187"> 
		///		<ExpectedInput>invalid division, bool</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a data table of units to be phenotyped
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="weakDTesting"></param>
		/// <returns></returns>
		public static DataTable GetPhenotypeUnits(string divisionCode, bool weakDTesting)
		{
			return DAL.BloodUnit.GetPhenotypeUnits(divisionCode,weakDTesting);
		}


		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/11/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1722"> 
		///		<ExpectedInput>Shipment GUID</ExpectedInput>
		///		<ExpectedOutput>data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1723"> 
		///		<ExpectedInput>Invalid shipment GUID</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns all blood units for a shipment
		/// </summary>
		/// <param name="shipmentGuid"></param>
		/// <returns></returns>
		public static DataTable GetBloodUnits(System.Guid shipmentGuid)
		{
			return DAL.BloodUnit.GetBloodUnits(shipmentGuid);
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5188"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5189"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a list of editable blood units for UC 70
		/// </summary>
		/// <param name="previouslyAssociatedUnits">Indicates whether to retrieve units that have a historical patient association</param>
		/// <returns>A list of blood units</returns>
		public static DataTable GetBloodUnitsForEdit(System.Boolean previouslyAssociatedUnits)
		{
			return DAL.BloodUnit.GetBloodUnitsForEdit(previouslyAssociatedUnits);
		}
		
		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/19/2007</CreationDate>
		///<Case type="0" testid ="8473"> 
		///		<ExpectedInput>Valid Shipment GUID</ExpectedInput>
		///		<ExpectedOutput>A string that contains all the units on the shipment and all the data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8474"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///
		/// <summary>
		/// Returns an array list of all units created on a shipment.  
		/// This is a static list of what units looked like upon initial entry in UC_01
		/// </summary>
		/// <param name="shipmentGuid"></param>
		/// <returns></returns>
		public static System.Collections.ArrayList GetIncomingShipmentBloodUnitsString(System.Guid shipmentGuid)
		{
			System.Data.DataTable dt = DAL.BloodUnit.GetBloodUnitsIncomingShipment(shipmentGuid);
			System.Collections.ArrayList alBloodUnits = new System.Collections.ArrayList();
						
			foreach (System.Data.DataRow dtRow in dt.Rows)
			{
				string values = (string)dtRow[TABLE.BloodUnitShipment.LoginValues];
				string[] subItems = values.Split('~');
                alBloodUnits.Add(subItems);
			}
			return alBloodUnits;
		}


		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/10/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3461"> 
		///		<ExpectedInput>Blood type code, int</ExpectedInput>
		///		<ExpectedOutput>The data row containing rh factor for the blood type</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3462"> 
		///		<ExpectedInput>Empty String, int</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets blood type and Rh factor information based on a blood type. Related to BR_1.20, TT_1.02
		/// </summary>
		/// <param name="bloodTypeCode">Blood type code</param>
		/// <returns>DataRow containing blood type and Rh factor data for a given blood type</returns>
		public static DataRow GetBloodTypeRHFactorFromBloodTypeCode(string bloodTypeCode)
		{
			return DAL.BloodUnit.GetBloodTypeRHFactorFromBloodTypeCode(bloodTypeCode);
		}   

		///<Developers>
		///	<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/24/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="741"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5213"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public method creates an empty data table for a blood unit
		/// </summary>
		public static System.Data.DataTable GetEmptyBloodUnitDataTable()
		{
			System.Data.DataTable dtBloodUnits = new System.Data.DataTable(TABLE.BloodUnit.TableName);
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.BloodUnitGuid, typeof(System.Guid));
			dtBloodUnits.Columns.Add(TABLE.RhFactor.RhFactorCode,typeof(String));		//new
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.DivisionCode,typeof(String));		//new - me
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.UnitContainerLotNumber, typeof(String));	//new
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.UnitDivisionId, typeof(String));	//new
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.BloodProductGuid,typeof(System.Guid));	//new
			dtBloodUnits.Columns.Add(TABLE.BloodUnitStatus.BiohazardousWarningIndicator,typeof(bool));	//new
			dtBloodUnits.Columns.Add(TABLE.BloodUnitStatus.UnitStatusComments,typeof(String));	//new
			dtBloodUnits.Columns.Add(TABLE.BloodUnitStatus.DiscardCannedCommentGuid,typeof(Guid));	//new
			dtBloodUnits.Columns.Add(TABLE.BloodUnitShipment.IncomingShipmentGuid,typeof(System.Guid));	//new
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.LabelTypeCode, typeof(String));		//new

			dtBloodUnits.Columns.Add(TABLE.BloodUnitMedia.BloodTypeCode, typeof(String));
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.EyeReadableUnitId, typeof(String));
			dtBloodUnits.Columns.Add(TABLE.Facility.FacilityName, typeof(String));
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.UnitFdaRegistrationNumber, typeof(String));
			dtBloodUnits.Columns.Add(TABLE.BloodUnitMedia.UnitExpirationDate, typeof(DateTime));
			dtBloodUnits.Columns.Add(TABLE.UnitStatus.UnitStatusText, typeof(String));
			dtBloodUnits.Columns.Add(TABLE.BloodUnitStatusCodeCurrent.UnitStatusCode,typeof(String));
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.DonationTypeId, typeof(int));
			dtBloodUnits.Columns.Add(TABLE.DonationType.DonationTypeText, typeof(String));
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.RowVersion, typeof(Byte[]));
			
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.UnitInDate, typeof(DateTime));
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.UnitProductCode, typeof(String));
			dtBloodUnits.Columns.Add(TABLE.ProductType.ProductTypeName, typeof(String));
			dtBloodUnits.Columns.Add(TABLE.BloodProduct.ProductName, typeof(String));
			dtBloodUnits.Columns.Add(TABLE.BloodProduct.ProductTypeCode, typeof(String));
			dtBloodUnits.Columns.Add(TABLE.BloodProduct.ProductShortName, typeof(String));
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.LabelUnitId, typeof(String));
			dtBloodUnits.Columns.Add(TABLE.BloodUnitMedia.DirectedPatientGuid, typeof(Guid));
			dtBloodUnits.Columns.Add(TABLE.BloodUnitMedia.TestingStatusId, typeof(String));

			dtBloodUnits.Columns.Add(TABLE.BloodUnit.OriginalVolume, typeof(decimal));
			dtBloodUnits.Columns.Add(TABLE.BloodProduct.MaximumStorageTime, typeof(decimal));
			dtBloodUnits.Columns.Add(TABLE.BloodUnitStatus.QuarantineIndicator, typeof(bool));
			dtBloodUnits.Columns.Add(TABLE.BloodUnitMedia.Comments, typeof(String));
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.LastUpdateDate,typeof(DateTime));
			dtBloodUnits.Columns.Add(TABLE.BloodUnit.RecordStatusCode, typeof(string));	//new me
			
			//MJ added row version columns for blood unit status, media, and finance, and blood unit row
			dtBloodUnits.Columns.Add(Common.DatabaseConstants.ArtificialColumnNames.UnitStatusRowVersion, typeof(Byte[]));
			dtBloodUnits.Columns.Add(Common.DatabaseConstants.ArtificialColumnNames.UnitMediaRowVersion, typeof(Byte[]));
			dtBloodUnits.Columns.Add(Common.DatabaseConstants.ArtificialColumnNames.UnitFinanceRowVersion, typeof(Byte[]));
			dtBloodUnits.Columns.Add(Common.DatabaseConstants.ArtificialColumnNames.UnitRowVersion, typeof(Byte[]));
			return dtBloodUnits;

		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4040"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4353"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns all blood units with status of final:
		/// discarded, modified, transfused or transferred
		/// </summary>
		/// <param name="isFullService">Indicates whether being called from a full-service or transfusion-only division</param>
		/// <returns>data table of blood units</returns>
		public static System.Data.DataTable GetBloodUnitsInFinalStatus(bool isFullService)
		{
			DataTable dtFinalStatus = DAL.BloodUnit.GetBloodUnitsInFinalStatus();
			//
			if (isFullService)
			{
				return dtFinalStatus;
			}
			//
			
			DataTable dtNonMods = dtFinalStatus.Clone();
			DataRow[] drNonModifieds = dtFinalStatus.Select(TABLE.BloodUnitStatusCodeCurrent.UnitStatusCode + "<> '"+Common.Utility.GetUnitStatusCodeCharFromEnum(Common.UnitStatusCode.Modified).ToString()+"'");
			if (drNonModifieds != null)
			{
				foreach(DataRow dr in drNonModifieds)
				{
					dtNonMods.Rows.Add(dr.ItemArray);
				}
			}
			return dtNonMods;
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5193"> 
		///		<ExpectedInput>Common.ComponentRequirement</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5194"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// IsComponentRequirementSatisfied - This returns a boolean indicating whether or not the unit
        /// satisfies this component requirement. 
        /// 
        /// // CR3260: Rewrote method based on new requirements.
        /// 
        /// Implements BR_3.14, BR_56.25. BR_73.28
        /// </summary>
        public bool IsComponentRequirementSatisfied(Common.ComponentRequirement compReq)
        {
            bool reqsMet = true;

            Common.ProductType productType = Common.Utility.GetProductTypeFromProductTypeCode(this.ProductType.ProductTypeCode);


            switch (compReq)
            {

                // IRRADIATED: Must hve C2 attribute
                case Common.ComponentRequirement.IRRADIATE:
                    // Applies to: 
                    //	RBC
                    //	WB
                    //	Platelets 
                    //	OTHER (all, NOT just cellular types)
                    if (this.BloodComponentClassId == (int)Common.ComponentClass.RBC ||
                        this.BloodComponentClassId == (int)Common.ComponentClass.WB ||
                        this.BloodComponentClassId == (int)Common.ComponentClass.PLT ||
                        (this.BloodComponentClassId == (int)Common.ComponentClass.Other)
                        )
                    {
                        // If it the blood unit does not have C2 then it is not valid
                        if (this.ProductAttributeForm.IndexOf(" C2") == -1)
                        {
                            reqsMet = false;
                        }
                    }
                    break;


                // LEUKOREDUCED: Must have an attribute of D*, not including D1
                case Common.ComponentRequirement.LEUKORED:
                    // Applies to: 
                    //	RBC
                    //	WB
                    //	Platelets
                    if (this.BloodComponentClassId == (int)Common.ComponentClass.RBC ||
                        this.BloodComponentClassId == (int)Common.ComponentClass.WB ||
                        this.BloodComponentClassId == (int)Common.ComponentClass.PLT)
                    {
                        if (this.ProductAttributeForm.IndexOf(" D") == -1 || // no D attributes
                        this.ProductAttributeForm.IndexOf(" D1") > -1) // D attribute, but its D1 (again, invalid)
                        {
                            reqsMet = false;
                        }
                    }
                    break;


                // WASHED RBC
                case Common.ComponentRequirement.WASHEDRBC:
                    if (this.BloodComponentClassId == (int)Common.ComponentClass.RBC ||
                        this.BloodComponentClassId == (int)Common.ComponentClass.WB)
                    {
                        if (productType != Common.ProductType.WashedRedBloodCells &&								// E003
                            productType != Common.ProductType.DeglycerolizedRedBloodCells &&					// E006
                            productType != Common.ProductType.DeglycerolizedRejuvenatedRedBloodCells &&			// E007
                            productType != Common.ProductType.WashedApheresisRedBloodCells &&					// E049
                            productType != Common.ProductType.DeglycerolizedApheresisRedBloodCells &&			// E051
                            productType != Common.ProductType.DeglycerolizedRejuvenatedApheresisRedBloodCells)	// E054		
                        {
                            reqsMet = false;
                        }
                    }
                    break;


                // WASHED PLATELET
                case Common.ComponentRequirement.WASHEDPLT:
                    if (this.BloodComponentClassId == (int)Common.ComponentClass.PLT)
                    {
                        if (productType != Common.ProductType.WashedPlatelets &&					// E021
                            productType != Common.ProductType.WashedPooledPlatelets &&			// E023
                            productType != Common.ProductType.WashedApheresisPlatelets)			// E027
                        {
                            reqsMet = false;
                        }
                    }
                    break;


                // SC NEGATIVE
                case Common.ComponentRequirement.SCNEG:
                    // Applies to: 
                    //	RBC
                    //	WB
                    if ((this.BloodComponentClassId == (int)Common.ComponentClass.RBC ||
                        this.BloodComponentClassId == (int)Common.ComponentClass.WB)
                        && this.SickleCellNegative == false)
                    {
                        reqsMet = false;
                    }
                    break;


                // CMV NEGATIVE
                case Common.ComponentRequirement.CMVNEG:
                    // Applies to: 
                    //	RBC
                    //	WB
                    //	Platelets
                    //	OTHER
                    if (this.BloodComponentClassId == (int)Common.ComponentClass.RBC ||
                        this.BloodComponentClassId == (int)Common.ComponentClass.WB ||
                        this.BloodComponentClassId == (int)Common.ComponentClass.PLT ||
                        this.BloodComponentClassId == (int)Common.ComponentClass.Other)
                    {
                        if (this.CmvNegative == false)
                        {
                            reqsMet = false;
                        }
                    }
                    break;
            }
            return reqsMet;
        }

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/23/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4053"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4348"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static function returns all the directed blood units that meet the requirements 
		/// of the following business rules: BR_71.01, BR_71.02 and BR_71.03
		/// </summary>
		/// <returns>DataTable</returns>
		public static DataTable GetBloodUnitsEligibleForCrossover()
		{
			return DAL.BloodUnit.GetBloodUnitsEligibleForCrossover();
		}


		#endregion

		#region Methods

		/// <summary>
		/// InstantiateAggregates
		/// </summary>
		/// <param name="bloodUnitDataRow"></param>
		private void InstantiateAggregates(System.Data.DataRow bloodUnitDataRow)
		{
			if (!this.IsNew && this._needsBloodProduct)
			{
				this.GetBloodProductByGuid(this.BloodProductGuid);
			}

			if(bloodUnitDataRow.Table.Columns.Contains(TABLE.BloodUnit.BloodUnitGuid))
			{
				this._bloodUnitStatus = new BOL.BloodUnitStatus(this.BloodUnitGuid);
				this._bloodUnitFinance = new BOL.BloodUnitFinance(this.BloodUnitGuid);
				this._bloodUnitMedia = new BOL.BloodUnitMedia(this.BloodUnitGuid);
			}

			if(bloodUnitDataRow.Table.Columns.Contains(TABLE.BloodUnit.UnitFdaRegistrationNumber))
			{
				if (!bloodUnitDataRow.IsNull(TABLE.BloodUnit.UnitFdaRegistrationNumber))
				{
					if (bloodUnitDataRow[TABLE.BloodUnit.UnitFdaRegistrationNumber].ToString().Trim() != String.Empty)
					{
						this._localSupplier = new BOL.LocalSupplier(bloodUnitDataRow[TABLE.BloodUnit.UnitFdaRegistrationNumber].ToString(),true);
					}
				}
			}

			if(bloodUnitDataRow.Table.Columns.Contains(TABLE.BloodUnit.DonationTypeId))
			{
				if (!bloodUnitDataRow.IsNull(TABLE.BloodUnit.DonationTypeId))
					this.DonationType = new BOL.DonationType(System.Convert.ToInt16(bloodUnitDataRow[TABLE.BloodUnit.DonationTypeId]));
			}

			if (this._bloodUnitMedia.DirectedPatientGuid != System.Guid.Empty)
			{
				this._directedPatient = new BOL.Patient(this._bloodUnitMedia.DirectedPatientGuid);
			}
		}

		
		/// <summary>
		/// MakeNew
		/// </summary>
		private void MakeNew()
		{
			if (_bloodUnitDataTable == null)
			{
				throw new NullReferenceException("BloodUnit DataTable is Null");
			}
			_bloodUnitDataRow = _bloodUnitDataTable.NewRow();
           
			_bloodUnitDataRow[TABLE.BloodUnit.BloodUnitGuid] = System.Guid.NewGuid();
			_bloodUnitDataRow[TABLE.BloodUnit.UnitProductCode] = string.Empty;
			_bloodUnitDataRow[TABLE.BloodUnit.LabelUnitId] = string.Empty;
			_bloodUnitDataRow[TABLE.BloodUnit.BloodProductGuid] = Guid.Empty;
			_bloodUnitDataRow[TABLE.BloodUnit.DivisionCode] = this.DivisionCode;
			_bloodUnitDataRow[TABLE.BloodUnit.LastUpdateDate] = VBECSDateTime.GetDivisionCurrentDateTime();
			_bloodUnitDataRow[TABLE.BloodUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
			_bloodUnitDataRow[TABLE.BloodUnit.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active);
			_bloodUnitDataRow[TABLE.BloodUnit.LabelTypeCode] = Common.Utility.GetBarcodeTypeCodeCharFromEnum(this.BarcodeType);
			if(_bloodUnitDataRow.Table.Columns.Contains(TABLE.BloodUnit.ProcessingTechId))
			{
				_bloodUnitDataRow[TABLE.BloodUnit.ProcessingTechId] = String.Empty;
			}
			if(_bloodUnitDataRow.Table.Columns.Contains(TABLE.DonationType.DonationTypeCode))
			{
				_bloodUnitDataRow[TABLE.DonationType.DonationTypeCode] = string.Empty;
			}

			_bloodUnitDataTable.Rows.Add(_bloodUnitDataRow);

			this.IsNew = true;
		}
        

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5195"> 
		///		<ExpectedInput>UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5196"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Common.RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Save
		/// </summary>
		/// <returns></returns>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		public new bool Save(Common.UpdateFunction lastUpdateFunctionId)
		{
			bool retval = false;

			System.Data.DataSet dsSaveBloodUnit = new DataSet();

			System.Data.DataTable dtBloodUnit = DAL.BloodUnit.GetEmptyBloodUnitTableSchema(true);
			System.Data.DataTable dtBloodUnitMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(true);
			System.Data.DataTable dtBloodUnitFinance = DAL.BloodUnitFinance.GetEmptyBloodUnitFinanceTableSchema(true);
			System.Data.DataTable dtBloodUnitStatus = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(true);

			dtBloodUnit.Rows.Add(this.LoadDataRowFromThis(dtBloodUnit.NewRow()));
			dtBloodUnitMedia.Rows.Add(this.BloodUnitMedia.LoadDataRowFromThis(dtBloodUnitMedia.NewRow()));
			dtBloodUnitFinance.Rows.Add(this.BloodUnitFinance.LoadDataRowFromThis(dtBloodUnitFinance.NewRow()));
			dtBloodUnitStatus.Rows.Add(this.BloodUnitStatus.LoadDataRowFromThis(dtBloodUnitStatus.NewRow()));

			dsSaveBloodUnit.Tables.Add(dtBloodUnit);
			dsSaveBloodUnit.Tables.Add(dtBloodUnitMedia);
			dsSaveBloodUnit.Tables.Add(dtBloodUnitFinance);
			dsSaveBloodUnit.Tables.Add(dtBloodUnitStatus);

			retval = DAL.BloodUnit.SaveBloodUnitAndAggregates(dsSaveBloodUnit, lastUpdateFunctionId);
			return retval;
		}

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3609"> 
		///		<ExpectedInput>OutgoingshipmentColletion, UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3618"> 
		///		<ExpectedInput>Invalid collection</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Save
		/// </summary>
		/// <returns></returns>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="shipment"></param>
		public bool SaveOutgoingShipment(BOL.OutgoingShipmentCollection shipment, Common.UpdateFunction lastUpdateFunctionId)
		{
			bool retval = false;

			System.Collections.ArrayList alShipmentTables =  BOL.BaseShipment.InitializeShipmentTable(false);
			
			DataTable dtBloodUnitShipment = (System.Data.DataTable) alShipmentTables[1];
			this.PopulateShipmentTables(dtBloodUnitShipment, shipment);

			System.Data.DataSet dsSaveBloodUnit = new DataSet();

			//12-15/2005 - Lohse - The only data that changes for OutgoingShipment is BloodUnitFinance --
			//					 - so -- removing the updates to all the other tables -- for locking purposes
			//			System.Data.DataTable dtBloodUnit = DAL.BloodUnit.GetEmptyBloodUnitTableSchema(true);
			//			System.Data.DataTable dtBloodUnitMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(true);
			System.Data.DataTable dtBloodUnitFinance = DAL.BloodUnitFinance.GetEmptyBloodUnitFinanceTableSchema(true);
			//			System.Data.DataTable dtBloodUnitStatus = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(true);

			//			dtBloodUnit.Rows.Add(this.LoadDataRowFromThis(dtBloodUnit.NewRow()));
			//			dtBloodUnitMedia.Rows.Add(this.BloodUnitMedia.LoadDataRowFromThis(dtBloodUnitMedia.NewRow()));
			dtBloodUnitFinance.Rows.Add(this.BloodUnitFinance.LoadDataRowFromThis(dtBloodUnitFinance.NewRow()));
			//			dtBloodUnitStatus.Rows.Add(this.BloodUnitStatus.LoadDataRowFromThis(dtBloodUnitStatus.NewRow()));

			//			dsSaveBloodUnit.Tables.Add(dtBloodUnit);
			//			dsSaveBloodUnit.Tables.Add(dtBloodUnitMedia);
			dsSaveBloodUnit.Tables.Add(dtBloodUnitFinance);
			//			dsSaveBloodUnit.Tables.Add(dtBloodUnitStatus);
			dsSaveBloodUnit.Tables.Add(dtBloodUnitShipment);

			retval = DAL.BloodUnit.SaveBloodUnitAndAggregates(dsSaveBloodUnit, lastUpdateFunctionId);

			return retval;
		}
		
	
		private void PopulateShipmentTables(DataTable dtBloodUnitShipment, BOL.OutgoingShipmentCollection shipment)
		{
			//Create a new Row in Shipment table and add data based on the information passed in
			DataRow drBloodUnitShipment = dtBloodUnitShipment.NewRow();

			//Populate the bloodunitshipment table
			drBloodUnitShipment[TABLE.BloodUnitShipment.BloodUnitShipmentGuid] = Guid.NewGuid();
			drBloodUnitShipment[TABLE.BloodUnitShipment.BloodUnitGuid] = this.BloodUnitGuid;

			if (shipment.ShipmentGuid == Guid.Empty)
			{
				drBloodUnitShipment[TABLE.BloodUnitShipment.OutgoingShipmentGuid] = Guid.NewGuid();
			}
			else
			{
				drBloodUnitShipment[TABLE.BloodUnitShipment.OutgoingShipmentGuid] = shipment.ShipmentGuid;
			}

			drBloodUnitShipment[TABLE.BloodUnitShipment.LastUpdateUser] = Common.LogonUser.LogonUserName;

			dtBloodUnitShipment.Rows.Add(drBloodUnitShipment);
		}

		///<Developers>
		///	<Developer>Martin Tsotchev</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3774"> 
		///		<ExpectedInput>Valid antigen array list, blood unit test array list and update function ID</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3775"> 
		///		<ExpectedInput>Empty antigen array list, blood unit test array list and update function ID. Also, setting the current blood unit's GUID and division code to invalid values.</ExpectedInput>
		///		<ExpectedOutput>False</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Updates certain blood unit fields, as well as blood unit antigens and
		/// sickle cell and CMV blood unit tests from within UC 70, Edit Unit Information
		/// Implements BR_70.08, BR_70.10, BR_70.15, BR_41.12
		/// </summary>
		/// <param name="updateBloodUnit">Whether to update the blood unit</param>
		/// <param name="removeAllAntigens">Tells if all antigens were removed</param>
		/// <param name="quaratineBloodUnit">Whether to quarantine the blood unit</param>
		/// <param name="antigens">Blood unit antigens</param>
		/// <param name="bloodUnitTests">Sickle cell and CMV blood unit tests</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns>Success/Failure</returns>
		public bool Save 
		(
			bool updateBloodUnit,
			bool removeAllAntigens, 
			bool quaratineBloodUnit, 
			ArrayList antigens, 
			ArrayList bloodUnitTests, 
			Common.UpdateFunction lastUpdateFunctionId
		)
		{
			DataTable[] dtArray = null;
			//
			// BR_70.08, BR_70.10, BR_70.15
			// First, create the table to hold the Blood Unit data
			// that needs to be updated
			DataTable dtBloodUnit = DAL.BloodUnit.GetEmptyBloodUnitTableSchema(true);
			DataTable dtBloodUnitMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(true);
			DataTable dtBloodUnitStatus = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(true);
			//
			// For quarantining the unit per BR_41.12
			DataTable dtWorkload = null;
			//
			dtBloodUnit.Rows.Add(this.LoadDataRowFromThis(dtBloodUnit.NewRow()));
			dtBloodUnitMedia.Rows.Add(this.BloodUnitMedia.LoadDataRowFromThis(dtBloodUnitMedia.NewRow()));
			dtBloodUnitStatus.Rows.Add(this.BloodUnitStatus.LoadDataRowFromThis(dtBloodUnitStatus.NewRow()));
			//
			// Next, create the table to hold the Blood Unit Antigen data
			DataTable dtInsertBloodUnitAntigens = BOL.BloodUnitAntigen.GetEmptyBloodUnitAntigenTable(false);
			DataTable dtUpdateBloodUnitAntigens = BOL.BloodUnitAntigen.GetEmptyBloodUnitAntigenTable(true);
			//
			foreach(BOL.BloodUnitAntigen antigen in antigens)
			{
				if(antigen.RowVersion == null)
				{
					System.Data.DataRow dtRow = antigen.LoadDataRowFromThis(dtInsertBloodUnitAntigens.NewRow());
					dtInsertBloodUnitAntigens.Rows.Add(dtRow);
				}
				else
				{
					System.Data.DataRow dtRow = antigen.LoadDataRowFromThis(dtUpdateBloodUnitAntigens.NewRow());
					if(dtUpdateBloodUnitAntigens.Rows.Contains(antigen.AntigenTypeId))
					{
						dtUpdateBloodUnitAntigens.Rows.Remove(dtUpdateBloodUnitAntigens.Rows.Find(antigen.AntigenTypeId));
					}
					dtUpdateBloodUnitAntigens.Rows.Add(dtRow);
				}
			}
			//
			// Last, create two tables to hold the Blood Unit Test data,
			// one for the inserts, one for the updates
			if (bloodUnitTests != null && bloodUnitTests.Count > 0)
			{
				DataTable dtBloodUnitTestsUpdate = new DataTable();
				dtBloodUnitTestsUpdate.Columns.Add(TABLE.BloodUnitTest.BloodUnitTestGuid, typeof(System.Guid));
				dtBloodUnitTestsUpdate.Columns.Add(TABLE.BloodUnitTest.TestResultId, typeof(string));
				dtBloodUnitTestsUpdate.Columns.Add(TABLE.BloodUnitTest.TestDate, typeof(DateTime));
				dtBloodUnitTestsUpdate.Columns.Add(TABLE.BloodUnitTest.TestTechId, typeof(string));
				dtBloodUnitTestsUpdate.Columns.Add(TABLE.BloodUnitTest.EntryTechId, typeof(string));
				dtBloodUnitTestsUpdate.Columns.Add(TABLE.BloodUnitTest.TestComments, typeof(string));
				dtBloodUnitTestsUpdate.Columns.Add(TABLE.BloodUnitTest.RowVersion, typeof(byte[]));
				dtBloodUnitTestsUpdate.Columns.Add(TABLE.BloodUnitTest.LastUpdateUser, typeof(string));
				dtBloodUnitTestsUpdate.Columns.Add(TABLE.BloodUnitTest.DivisionCode, typeof(string));
				//
				DataTable dtBloodUnitTestsAdd = dtBloodUnitTestsUpdate.Clone();
				dtBloodUnitTestsAdd.Columns.Add(TABLE.BloodUnitTest.BloodUnitGuid, typeof(System.Guid));
				dtBloodUnitTestsAdd.Columns.Add(TABLE.BloodUnitTest.BloodTestTypeId, typeof(int));
				dtBloodUnitTestsAdd.Columns.Add(TABLE.BloodUnitTest.EntryMethodCode, typeof(string));
				//
				foreach(BOL.BloodUnitTest bloodUnitTest in bloodUnitTests)
				{
					if (bloodUnitTest.IsNew)
					{
						DataRow dr = dtBloodUnitTestsAdd.NewRow();
						dr[TABLE.BloodUnitTest.BloodUnitTestGuid] = Guid.NewGuid();
						dr[TABLE.BloodUnitTest.BloodUnitGuid] = this.BloodUnitGuid;
						dr[TABLE.BloodUnitTest.BloodTestTypeId] = bloodUnitTest.BloodTestTypeObj.BloodTestTypeId;
						dr[TABLE.BloodUnitTest.TestResultId] = bloodUnitTest.TestResultId;
						dr[TABLE.BloodUnitTest.TestDate] = VBECSDateTime.GetDivisionCurrentDateTime();
						dr[TABLE.BloodUnitTest.TestTechId] = this.LastUpdateUser;
						dr[TABLE.BloodUnitTest.EntryMethodCode] = "M";
						dr[TABLE.BloodUnitTest.EntryTechId] = this.LastUpdateUser;
						dr[TABLE.BloodUnitTest.TestComments] = bloodUnitTest.TestComments;
						dr[TABLE.BloodUnitTest.RowVersion] = DBNull.Value;
						dr[TABLE.BloodUnitTest.LastUpdateUser] = this.LastUpdateUser;
						dr[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
						dtBloodUnitTestsAdd.Rows.Add(dr);
					}
					else
					{	
						DataRow dr = dtBloodUnitTestsUpdate.NewRow();
						dr[TABLE.BloodUnitTest.BloodUnitTestGuid] = bloodUnitTest.BloodUnitTestGuid;
						dr[TABLE.BloodUnitTest.TestResultId] = bloodUnitTest.TestResultId;
						dr[TABLE.BloodUnitTest.TestDate] = VBECSDateTime.GetDivisionCurrentDateTime();
						dr[TABLE.BloodUnitTest.TestTechId] = this.LastUpdateUser;
						dr[TABLE.BloodUnitTest.EntryTechId] = this.LastUpdateUser;
						dr[TABLE.BloodUnitTest.TestComments] = bloodUnitTest.TestComments;
						dr[TABLE.BloodUnitTest.RowVersion] = bloodUnitTest.RowVersion;
						dr[TABLE.BloodUnitTest.LastUpdateUser] = this.LastUpdateUser;
						dr[TABLE.BloodUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
						dtBloodUnitTestsUpdate.Rows.Add(dr);
					}
				}
				//
				dtArray = new DataTable[2];
				dtArray[0] = dtBloodUnitTestsAdd;
				dtArray[1] = dtBloodUnitTestsUpdate;
			}
			// BR_41.12
			if(quaratineBloodUnit)
			{
				dtBloodUnitStatus.Rows[0][TABLE.BloodUnitStatus.QuarantineCannedCommentGuid] = DBNull.Value;
				dtBloodUnitStatus.Rows[0][TABLE.BloodUnitStatus.QuarantineIndicator] = 1;
				dtBloodUnitStatus.Rows[0][TABLE.BloodUnitStatus.UnitStatusComments] = Common.StrRes.InfoMsg.UC070.InconsistentAntigenType().ResString;
			}
			//
			// Now, execute the actual save
			return DAL.BloodUnit.Save
			(
				dtBloodUnit, 
				removeAllAntigens, 
				dtBloodUnitMedia, 
				dtBloodUnitStatus, 
				dtInsertBloodUnitAntigens, 
				dtUpdateBloodUnitAntigens, 
				dtWorkload,
				dtArray, 
				lastUpdateFunctionId	
			);
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2589"> 
		///		<ExpectedInput>Valid alphanumeric unit ID</ExpectedInput>
		///		<ExpectedOutput>Unit ID converted to all numbers</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2590"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// If a Codebar Unit ID is manually entered, any letters that may have been in this field 
		/// need to be converted to numbers.  The code below reverses the logic in TT 1.01 from I2. TT_1.04
		/// </summary>
		/// 
		/// BR_1.05
		public string GenerateNumericCodabarUnitID(String unitID, ref bool alphaCharactersPresent)
		{
			string lastChar = unitID.Substring(unitID.Length - 1);
 
			StringBuilder alphaCharacters = new StringBuilder();
			int alphaCount = 0;		
			int newNumber = 0;
			alphaCharactersPresent = false;

			if (char.IsLetter(lastChar, 0)) 
			{
				unitID = unitID.Substring(0, unitID.Length-1);
			}
			if(this._localSupplier != null)
			{
				if(this._localSupplier.FDARegistrationNumber.Length != 0)
				{
					if(this._localSupplier.FacilityPrefix.Length ==2)
					{
						unitID = unitID.Substring(2);
					}
					if(unitID.Length > 7)
					{
						unitID = unitID.Substring(0, 7);
					}
				}
			}
						
			for (int i=0; i<unitID.Length; i++)
			{
				if (char.IsLetter(unitID, i)) 
				{
					alphaCharactersPresent = true;
					alphaCharacters.Append(unitID.Substring(i,1));
					alphaCount++;
				}
				// Create the new unit ID
				if (char.IsDigit(unitID, i))
				{
					if (alphaCharactersPresent == true)
					{
						unitID = unitID.Substring(i);
						
						// 1 alpha character
						if (alphaCount == 1)
						{
							newNumber = "CEFGHJKLMNPQRSTVWXYZ".IndexOf(alphaCharacters.ToString().ToUpper());
							if(newNumber == -1)
							{
								throw new BusinessObjectException("Invalid Codabar Unit ID");
							}
							if (newNumber < 10)
							{
								unitID = "0" + newNumber + unitID;
							}
							else
							{
								unitID = newNumber + unitID;
							}

						}
							// 2 alpha characters
						else
						{
							if(alphaCharacters.Length >= 1)
							{
								newNumber = "CEFGHJKLMNPQRSTVWXYZ".IndexOf(alphaCharacters.ToString().Substring(1,1).ToUpper());
								if(newNumber == -1)
								{
									throw new BusinessObjectException("Invalid Codabar Unit ID");
								}
								if (newNumber>9) newNumber = newNumber - 10;


								if ( "CEFGHJKLMN".IndexOf(alphaCharacters.ToString().Substring(1,1).ToUpper()) >= 0 )
								{
									switch (alphaCharacters.ToString().Substring(0,1).ToUpper())

									{
										case "F":
											unitID = "2" + newNumber + unitID;
											break;
										case "G":
											unitID = "4" + newNumber + unitID;
											break;
										case "K":
											unitID = "6" + newNumber + unitID;
											break;
										case "L":
											unitID = "8" + newNumber + unitID;
											break;
										default:
											throw new BusinessObjectException("Invalid Codabar Unit ID");
									}
								}
								else
								{
									switch (alphaCharacters.ToString().Substring(0,1).ToUpper())
									{
										case "F":
											unitID = "3" + newNumber + unitID;
											break;
										case "G":
											unitID = "5" + newNumber + unitID;
											break;
										case "K":
											unitID = "7" + newNumber + unitID;
											break;
										case "L":
											unitID = "9" + newNumber + unitID;
											break;
										default:
											throw new BusinessObjectException("Invalid Codabar Unit ID");
									}
								}
							}
						} // if alphaCount...
						break;
					} // alphaChars present 
				} //  if char isDigit
			} // for	
			return unitID;
		}


		/// <summary>
		/// GetAssignedToPatients
		/// </summary>
		private void GetAssignedToPatients()
		{
			this._assignedToPatients = new System.Collections.ArrayList();

			DataTable dt = DAL.PatientAssignments.GetPatientsAssignmentDetails(this.BloodUnitGuid);
			
			foreach(DataRow dr in dt.Rows)
			{
				this._assignedToPatients.Add(new Patient(dr));
			}

		}


		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5197"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5198"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetRBCAntigens
		/// </summary>
		/// <returns></returns>
		public System.Data.DataTable GetRbcAntigens()
		{
			return BOL.BloodUnitAntigen.GetBloodUnitAntigens(this.BloodUnitGuid, this.DivisionCode);
		}

		
		/// <summary>
		/// LoadFromDataRow
		/// </summary>
		/// <param name="dtRow"></param>
		protected new void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			char donationTypeCode = char.MinValue;
			 
			if (dtRow.Table.Columns.Contains(TABLE.BloodUnit.BloodUnitGuid))
			{
				if(dtRow.IsNull(TABLE.BloodUnit.BloodUnitGuid) || dtRow[TABLE.BloodUnit.BloodUnitGuid].ToString().Trim().Length ==0)
				{
					if(this.BloodUnitGuid != System.Guid.Empty)
					{
						this.BloodUnitGuid = System.Guid.Empty;
					}
				}
				else if(((System.Guid)(dtRow[TABLE.BloodUnit.BloodUnitGuid])) != this.BloodUnitGuid)
				{
					this.BloodUnitGuid = (System.Guid)(dtRow[TABLE.BloodUnit.BloodUnitGuid]);
				}
			}	
			if (dtRow.Table.Columns.Contains(TABLE.BloodUnit.EyeReadableUnitId))
			{
				if(dtRow.IsNull(TABLE.BloodUnit.EyeReadableUnitId))
				{
					if(this.EyeReadableUnitId != String.Empty)
					{
						this.SetEyeReadableUnitId(string.Empty, false);
					}
				}
				else if(((string)(dtRow[TABLE.BloodUnit.EyeReadableUnitId])) != this.EyeReadableUnitId)
				{
					this.SetEyeReadableUnitId((string)(dtRow[TABLE.BloodUnit.EyeReadableUnitId]), false);
				}
			}
			if (dtRow.Table.Columns.Contains(TABLE.BloodUnitMedia.DirectedPatientGuid))
			{
				if(dtRow.IsNull(TABLE.BloodUnitMedia.DirectedPatientGuid) == false && dtRow[TABLE.BloodUnitMedia.DirectedPatientGuid].ToString().Trim().Length !=0)
				{
					if ((System.Guid) dtRow[TABLE.BloodUnitMedia.DirectedPatientGuid] == Guid.Empty)
					{
						this._directedPatient = new BOL.Patient();
					}
					else
					{
						this._directedPatient = new BOL.Patient((System.Guid) dtRow[TABLE.BloodUnitMedia.DirectedPatientGuid]);
					}
				}
				else
				{
					this._directedPatient = new BOL.Patient();
				}
			}

			
			if (dtRow.Table.Columns.Contains(TABLE.BloodUnit.DonationTypeId))
			{
				if(!dtRow.IsNull(TABLE.BloodUnit.DonationTypeId))
				{
					donationTypeCode = BOL.DonationType.GetDonationTypeCodeById((int) dtRow[TABLE.BloodUnit.DonationTypeId]);
					if(this.DonationType == null)
					{
						this.DonationType = new BOL.DonationType(donationTypeCode);
					}
					else
					{
						this.DonationType.DonationTypeCode = donationTypeCode;
					}
				}
			}
			

			if (dtRow.Table.Columns.Contains(TABLE.BloodUnit.UnitProductCode))
			{
				if(dtRow.IsNull(TABLE.BloodUnit.UnitProductCode))
				{
					if(this.ProductCode != String.Empty)
					{
						this.ProductCode = String.Empty;
					}
				}
				else if(((string)(dtRow[TABLE.BloodUnit.UnitProductCode])) != this.ProductCode)
				{
					if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.LabelTypeCode) && !dtRow.IsNull(TABLE.BloodUnit.LabelTypeCode))
					{
						//ISBT
						if(dtRow[TABLE.BloodUnit.LabelTypeCode].ToString() == "I")
						{
							this.BarcodeType = Common.BarcodeType.ISBT128;
							if(donationTypeCode != char.MinValue && dtRow.Table.Columns.Contains(TABLE.BloodUnit.UnitDivisionId))
							{
								this.ProductCode = (string)(dtRow[TABLE.BloodUnit.UnitProductCode]) + donationTypeCode + (string) (dtRow[TABLE.BloodUnit.UnitDivisionId]);
							}
							else if(donationTypeCode != char.MinValue)
							{
								this.ProductCode = (string)(dtRow[TABLE.BloodUnit.UnitProductCode]) + donationTypeCode;
							}
							else
							{
								this.ProductCode = (string)(dtRow[TABLE.BloodUnit.UnitProductCode]);
							}
						}
					}
					else
					{
						this.ProductCode = (string)(dtRow[TABLE.BloodUnit.UnitProductCode]);
					}

				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.LabelTypeCode) && !dtRow.IsNull(TABLE.BloodUnit.LabelTypeCode))
			{
				//Codabar
				if(dtRow[TABLE.BloodUnit.LabelTypeCode].ToString() == "C")
				{
					this.BarcodeType = Common.BarcodeType.Codabar;
				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.RhFactor.RhFactorCode))
			{
				if(this._bloodUnitMedia != null)
				{
					if (!dtRow.IsNull(TABLE.RhFactor.RhFactorCode))
					{
						this._bloodUnitMedia.AboRh.RH = AboRh.GetRHFromCode(dtRow[TABLE.RhFactor.RhFactorCode].ToString());
					}
				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitMedia.UnitExpirationDate))
			{
				if(dtRow.IsNull(TABLE.BloodUnitMedia.UnitExpirationDate))
				{
					if(this.BloodUnitMedia.UnitExpirationDate != System.DateTime.MinValue)
					{
						this.BloodUnitMedia.UnitExpirationDate = System.DateTime.MinValue;
					}
				}
				else if(((System.DateTime)(dtRow[TABLE.BloodUnitMedia.UnitExpirationDate])) != this.BloodUnitMedia.UnitExpirationDate)
				{
					this.BloodUnitMedia.UnitExpirationDate = (System.DateTime)(dtRow[TABLE.BloodUnitMedia.UnitExpirationDate]);
				}
			}
		
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.UnitContainerLotNumber))
			{
				if(dtRow.IsNull(TABLE.BloodUnit.UnitContainerLotNumber))
				{
					if(this.UnitContainerLotNumber != string.Empty)
					{
						this.UnitContainerLotNumber = String.Empty;
					}
				}
				else if(((string)(dtRow[TABLE.BloodUnit.UnitContainerLotNumber])) != this.UnitContainerLotNumber)
				{
					this.UnitContainerLotNumber = (string)(dtRow[TABLE.BloodUnit.UnitContainerLotNumber]);
				}
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.UnitDivisionId))
			{
				if(dtRow.IsNull(TABLE.BloodUnit.UnitDivisionId))
				{
					if(this.UnitDivisionId != String.Empty)
					{
						this.UnitDivisionId = String.Empty;
					}
				}
				else if(((string)(dtRow[TABLE.BloodUnit.UnitDivisionId])) != this.UnitDivisionId)
				{
					this.UnitDivisionId = (string)(dtRow[TABLE.BloodUnit.UnitDivisionId]);
				}
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.BloodProductGuid))
			{
				if(!dtRow.IsNull(TABLE.BloodUnit.BloodProductGuid))
				{
					this.BloodProductGuid = (System.Guid)dtRow[TABLE.BloodUnit.BloodProductGuid];
				}
			}


			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.UnitInDate))
			{
				if(dtRow.IsNull(TABLE.BloodUnit.UnitInDate))
				{
					if(this.UnitInDate != System.DateTime.MinValue)
					{
						this.UnitInDate = System.DateTime.MinValue;
					}
				}
				else if((System.DateTime)(dtRow[TABLE.BloodUnit.UnitInDate]) != this.UnitInDate)
				{
					this.UnitInDate = (System.DateTime)(dtRow[TABLE.BloodUnit.UnitInDate]);
				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitMedia.BloodTypeCode))
			{
				if(dtRow.IsNull(TABLE.BloodUnitMedia.BloodTypeCode))
				{
					if(this.BloodUnitMedia.BloodTypeCode != String.Empty)
					{
						this.BloodUnitMedia.BloodTypeCode = String.Empty;
					}
				}
				else if((string)(dtRow[TABLE.BloodUnitMedia.BloodTypeCode]) != this.BloodUnitMedia.BloodTypeCode)
				{
					this.BloodUnitMedia.BloodTypeCode = (string)(dtRow[TABLE.BloodUnitMedia.BloodTypeCode]);
				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.UnitFdaRegistrationNumber))
			{
				if(dtRow.IsNull(TABLE.BloodUnit.UnitFdaRegistrationNumber))
				{
					if(this.UnitFdaRegistrationNumber != String.Empty)
					{
						this.UnitFdaRegistrationNumber = string.Empty;
					}
				}
				else if(this.UnitFdaRegistrationNumber != (string)(dtRow[TABLE.BloodUnit.UnitFdaRegistrationNumber]))
				{
					this.UnitFdaRegistrationNumber = (string)(dtRow[TABLE.BloodUnit.UnitFdaRegistrationNumber]);
				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.LabelUnitId))
			{
				if(dtRow.IsNull(TABLE.BloodUnit.LabelUnitId))
				{
					if(this.LabelUnitId != String.Empty)
					{
						this.SetLabelUnitId(string.Empty, false);
					}
				}
				else if(this.LabelUnitId != (string)(dtRow[TABLE.BloodUnit.LabelUnitId]))
				{
					this.SetLabelUnitId((string)(dtRow[TABLE.BloodUnit.LabelUnitId]), false);
				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodProduct.ProductTypeCode))
			{
				if(this.ProductType == null)
				{
					this.ProductType = new BOL.ProductType();
				}
				if(dtRow.IsNull(TABLE.BloodProduct.ProductTypeCode))
				{
					if(this.ProductType.ProductTypeCode != String.Empty)
					{
						this.ProductType.ProductTypeCode = String.Empty;
					}
				}
				else if(this.ProductType.ProductTypeCode != (string)(dtRow[TABLE.BloodProduct.ProductTypeCode]))
				{
					this.ProductType.ProductTypeCode = (string)(dtRow[TABLE.BloodProduct.ProductTypeCode]);
				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodProduct.ProductName))
			{
				if(dtRow.IsNull(TABLE.BloodProduct.ProductName))
				{
					if(this.ProductName != String.Empty)
					{
						this.ProductName = String.Empty;
					}
				}
				else if(this.ProductName != (string)(dtRow[TABLE.BloodProduct.ProductName]))
				{
					this.ProductName = (string)(dtRow[TABLE.BloodProduct.ProductName]);
				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodProduct.ProductShortName))
			{
				if(dtRow.IsNull(TABLE.BloodProduct.ProductShortName))
				{
					if(this.ProductShortName != String.Empty)
					{
						this.ProductShortName = String.Empty;
					}
				}
				else if(this.ProductShortName != (string)(dtRow[TABLE.BloodProduct.ProductShortName]))
				{
					this.ProductShortName = (string)(dtRow[TABLE.BloodProduct.ProductShortName]);
				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.ProductType.ProductTypeName))
			{
				if(this.ProductType == null)
				{
					this.ProductType = new BOL.ProductType();
				}
				if(dtRow.IsNull(TABLE.ProductType.ProductTypeName))
				{
					if(this.ProductType.ProductTypeText != String.Empty)
					{
						this.ProductType.ProductTypeText = String.Empty;
					}
				}
				else if(this.ProductType.ProductTypeText != (string)(dtRow[TABLE.ProductType.ProductTypeName]))
				{
					this.ProductType.ProductTypeText = dtRow.IsNull(TABLE.ProductType.ProductTypeName) ? String.Empty : (string)(dtRow[TABLE.ProductType.ProductTypeName]);
				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodProduct.MaximumStorageTime))
			{
				if(dtRow.IsNull(TABLE.BloodProduct.MaximumStorageTime))
				{
					if(this.MaximumStorageDays != int.MinValue)
					{
						this.MaximumStorageDays = 0;
					}
					if(this.MaximumStorageHours != int.MinValue)
					{
						this.MaximumStorageHours = 0;
					}
				}
				else if(this.MaximumStorageDays != Convert.ToInt32((Convert.ToDecimal(dtRow[TABLE.BloodProduct.MaximumStorageTime])/24)))
				{
					this.MaximumStorageDays = Convert.ToInt32(Convert.ToDecimal(dtRow[TABLE.BloodProduct.MaximumStorageTime])/24);
				}
				else if(this.MaximumStorageHours != Convert.ToInt32(Convert.ToDecimal(dtRow[TABLE.BloodProduct.MaximumStorageTime])%24))
				{
					this.MaximumStorageHours = Convert.ToInt32(Convert.ToDecimal(dtRow[TABLE.BloodProduct.MaximumStorageTime])%24);
				}
			}        	
			if (dtRow.Table.Columns.Contains(TABLE.BloodUnit.LabelTypeCode))
			{
				if (dtRow.IsNull(TABLE.BloodUnit.LabelTypeCode))
				{
					this.BarcodeType = Common.BarcodeType.None;
				}
				else
				{
					this.BarcodeType = Common.Utility.GetBarcodeTypeFromString(dtRow[TABLE.BloodUnit.LabelTypeCode].ToString());
				}
			}
			else
			{
				this.BarcodeType = Common.BarcodeType.None;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitMedia.TestingStatusId))
			{
				if(dtRow.IsNull(TABLE.BloodUnitMedia.TestingStatusId))
				{
					if(this.BloodUnitMedia.TestingStatusId != int.MinValue)
					{
						this.BloodUnitMedia.TestingStatusId = int.MinValue;
					}
				}
				else if(this.BloodUnitMedia.TestingStatusId != Convert.ToInt32(dtRow[TABLE.BloodUnitMedia.TestingStatusId]))
				{
					this.BloodUnitMedia.TestingStatusId = Convert.ToInt32(dtRow[TABLE.BloodUnitMedia.TestingStatusId]);
				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.OriginalVolume))
			{
				if(dtRow.IsNull(TABLE.BloodUnit.OriginalVolume))
				{
					if(this.OriginalVolume != decimal.MinValue)
					{
						this.OriginalVolume = decimal.MinValue;
					}
				}
				else if(this.OriginalVolume != Convert.ToDecimal(dtRow[TABLE.BloodUnit.OriginalVolume]))
				{
					this.OriginalVolume = Convert.ToDecimal(dtRow[TABLE.BloodUnit.OriginalVolume]);
				}
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitMedia.Comments))
			{
				if(dtRow.IsNull(TABLE.BloodUnitMedia.Comments))
				{
					if(this.BloodUnitMedia.Comments != string.Empty)
					{
						this.BloodUnitMedia.Comments = string.Empty;
					}
				}
				else if(this.BloodUnitMedia.Comments != (string)(dtRow[TABLE.BloodUnitMedia.Comments]))
				{
					this.BloodUnitMedia.Comments = (string)(dtRow[TABLE.BloodUnitMedia.Comments]);
				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitShipment.IncomingShipmentGuid))
			{		
				if(dtRow.IsNull(TABLE.BloodUnitShipment.IncomingShipmentGuid))
				{
					this.IncomingShipmentGuid = System.Guid.Empty;
				}
				else
				{
					if(this.IncomingShipmentGuid !=(System.Guid)dtRow[TABLE.BloodUnitShipment.IncomingShipmentGuid])
					{
						this.IncomingShipmentGuid = (System.Guid)dtRow[TABLE.BloodUnitShipment.IncomingShipmentGuid];
						this.IncomingShipment = new BOL.IncomingShipment(this.IncomingShipmentGuid);
					}
				}
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.ProcessingTechId))
			{
				if(dtRow.IsNull(TABLE.BloodUnit.ProcessingTechId))
				{
					this.ProcessingTechId = string.Empty;
				}
				else
				{
					this.ProcessingTechId = dtRow[TABLE.BloodUnit.ProcessingTechId].ToString();
				}
			}
			//MJ added row version columns for blood unit
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.RowVersion))
			{
				if(!dtRow.IsNull(TABLE.BloodUnit.RowVersion))
				{
					this.RowVersion = (byte[]) dtRow[TABLE.BloodUnit.RowVersion];
				}
			}
		}

        
		/// <summary>
		/// SetEvents
		/// </summary>
		protected void SetEvents()
		{
			this._bloodUnitMedia.TestingStatusChanged += new EventHandler(this.BiohazardousWarningChanged);
			this.DonationType.DonationTypeCodeChanged += new EventHandler(this.BiohazardousWarningChanged);
			this.DonationType.TestingStatusChanged += new EventHandler(this.BiohazardousWarningChanged);
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/12/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1753"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1754"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean showing if the blood unit contains red blood cells
		/// </summary>
		/// <returns></returns>
		public bool ContainsRedBloodCells()
		{
			switch (Common.Utility.GetProductTypeFromProductTypeCode(this.ProductType.ProductTypeCode))
			{
				case (Common.ProductType.WholeBlood):
				case (Common.ProductType.RedBloodCells):
				case (Common.ProductType.WashedRedBloodCells):
				case (Common.ProductType.FrozenRedBloodCells):
				case (Common.ProductType.FrozenRejuvenatedRedBloodCells):
				case (Common.ProductType.DeglycerolizedRedBloodCells):
				case (Common.ProductType.DeglycerolizedRejuvenatedRedBloodCells):
				case (Common.ProductType.RejuvenatedRedBloodCells):
				case (Common.ProductType.ApheresisRedBloodCells):
				case (Common.ProductType.WashedApheresisRedBloodCells):
				case (Common.ProductType.FrozenApheresisRedBloodCells):
				case (Common.ProductType.DeglycerolizedApheresisRedBloodCells):
				case (Common.ProductType.RejuvenatedApheresisRedBloodCells):
				case (Common.ProductType.FrozenRejuvenatedApheresisRedBloodCells):
				case (Common.ProductType.DeglycerolizedRejuvenatedApheresisRedBloodCells):
					return true;
				default:
					return false;
			}
		}


		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/23/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1463"> 
		///		<ExpectedInput>Codabar unit which has a LabelUnitId greater than zero</ExpectedInput>
		///		<ExpectedOutput>Valid eye readable unit ID</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1464"> 
		///		<ExpectedInput>Codabar unit which has a LabelUnitId of zero</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generate EyeReadableUnitId from the label unit ID, TT_1.01
		/// </summary>
		public void GenerateEyeReadableUnitID()
		{
			if ((this.LabelUnitId.Length > 0) && (this.BarcodeType == Common.BarcodeType.Codabar))
			{
				//
				if (
					((object) _localSupplier!=null) && 
					(_localSupplier.FDARegistrationNumber!=null) && 
					(_localSupplier.FDARegistrationNumber.Length > 0))
				{
					string _unitID,_unitID2;
					int _intUnitID;
					//
					string _returnBloodUnitID = this.LabelUnitId;
					try
					{
					    //BR 1.16
						if (this._localSupplier.LabelUnitIDTrimIndicator)
						{
							_unitID=_returnBloodUnitID.Substring(0,2);
							_intUnitID = Int16.Parse(_unitID)/20;
							_unitID = ((Int16.Parse(_unitID) % 20) + 1).ToString();
							_unitID2= (_intUnitID == 0) ? String.Empty : "FGKL".Substring(_intUnitID - 1,1);
							_unitID = "CEFGHJKLMNPQRSTVWXYZ".Substring(Int16.Parse(_unitID) - 1,1);
							_returnBloodUnitID = (_unitID2 + _unitID) + _returnBloodUnitID.Substring(2,5);

							//BR 1.17
							if (this._localSupplier.FacilityPrefix.Trim().Length > 0)
							{
								_returnBloodUnitID = this._localSupplier.FacilityPrefix.Trim() + _returnBloodUnitID;
							}
						}
					}
					catch 
					{
						RuleBroken("LabelUnitId", true);
						throw new BOL.BusinessObjectException("Invalid codabar unit id");
					}
					this.SetEyeReadableUnitId(_returnBloodUnitID, true);
				}
			}
			else if ((this.LabelUnitId.Length > 0) && (this.BarcodeType == Common.BarcodeType.ISBT128))
			{
				this.SetEyeReadableUnitId(this.LabelUnitId, true);
			}
			else
			{
				throw new BOL.BusinessObjectException("Invalid barcode type or unit id");
			}
			
		}
		
		
		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5201"> 
		///		<ExpectedInput>localsupplier, unitid</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5202"> 
		///		<ExpectedInput>invalid parameters</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///<Case type="0" testid ="8179"> 
		///		<ExpectedInput>Codabar unit which has a UnitId like "test"</ExpectedInput>
		///		<ExpectedOutput>BusinessObjectException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GenerateEyeReadableUnitID  //Used for Codabar units only
		/// </summary>
		/// <param name="localSupplier"></param>
		/// <param name="unitID"></param>
		/// <returns></returns>
		public static string GenerateEyeReadableUnitID(BOL.LocalSupplier localSupplier, string unitID)
		{
			if ((unitID.Length > 0))
			{
				//
				if (
					((object) localSupplier!=null) && 
					(localSupplier.FDARegistrationNumber!=null) && 
					(localSupplier.FDARegistrationNumber.Length > 0))
				{
					string _unitID,_unitID2;
					//long _longUnitID;
					int _intUnitID;
					//
					
					try
					{
						//BR 1.16
						if (localSupplier.LabelUnitIDTrimIndicator)
						{
							_unitID=unitID.Substring(0,2);
							_intUnitID = Int16.Parse(_unitID)/20;
							_unitID = ((Int16.Parse(_unitID) % 20) + 1).ToString();
							_unitID2= (_intUnitID == 0) ? String.Empty : "FGKL".Substring(_intUnitID - 1,1);
							_unitID = "CEFGHJKLMNPQRSTVWXYZ".Substring(Int16.Parse(_unitID) - 1,1);
							unitID = (_unitID2 + _unitID) + unitID.Substring(2,5);

							//BR 1.17
							if (localSupplier.FacilityPrefix.Trim().Length > 0)
							{
								unitID = localSupplier.FacilityPrefix.Trim() + unitID;
							}
						}
					}
					catch 
					{
						throw new BOL.BusinessObjectException("Invalid codabar unit id");
					}
				}
			}
			return unitID;			
		}


		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5203"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5204"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Emtpy DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetBloodUnitByGuid
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public System.Data.DataTable GetBloodUnitByGuid(System.Guid bloodUnitGuid)
		{
			return DAL.BloodUnit.GetBloodUnitByGuid(bloodUnitGuid);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4041"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4354"> 
		///		<ExpectedInput>Invalid guid</ExpectedInput>
		///		<ExpectedOutput>Empty string</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the children/split blood units of this parent
		/// Note: see BR_13.05 for applicability
		/// </summary>
		/// <param name="bloodUnitGuid">Blood Unit ID</param>
		/// <returns>string consisting of child/split blood units eye-readable unit ID and product short name</returns>
		public string GetChildBloodUnitMessageString(Guid bloodUnitGuid)
		{
			StringBuilder message = new StringBuilder();

			if (this.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Modified)
			{
				System.Data.DataTable dtChildren = DAL.BloodUnitModification.GetModifiedBloodUnitChildren(bloodUnitGuid);
				if (dtChildren.Rows.Count > 0)
				{
					foreach(System.Data.DataRow dr in dtChildren.Rows)
					{
						message.Append("\n");
						message.Append(dr[TABLE.BloodUnit.EyeReadableUnitId].ToString());
						message.Append(", ");
						message.Append(dr[TABLE.BloodProduct.ProductShortName].ToString());
					}
				}
			}
			return message.ToString();
		}


		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4042"> 
		///		<ExpectedInput>Unit Guid</ExpectedInput>
		///		<ExpectedOutput>Object[]</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4355"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the last valid UnitStatus in both text and code forms
		/// for the passed-in blood unit id.
		/// </summary>
		/// <param name="bloodUnitGuid">Blood Unit ID</param>
		/// <returns>array consisting of: [0] unit status text (string) [1] unit status code (object)</returns>
		public Object[] GetPreviousBloodUnitStatusTextAndCodeByGuid(Guid bloodUnitGuid)
		{
			Object [] statusTextAndCode = new Object[2];

			System.Data.DataTable dtStatus = DAL.BloodUnitStatus.GetPreviousBloodUnitStatusByGuid(bloodUnitGuid);
			if (dtStatus.Rows.Count == 1)
			{
				statusTextAndCode[0] = (dtStatus.Rows[0][TABLE.UnitStatus.UnitStatusText].ToString());
				statusTextAndCode[1] = (Common.Utility.GetUnitStatusCodeFromString(dtStatus.Rows[0][TABLE.UnitStatus.UnitStatusCode].ToString()));
			}

			return statusTextAndCode;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4043"> 
		///		<ExpectedInput>UnitstatusCode, Comment, WorkloadProcessID</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4356"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Rolls back final status (discarded)
		/// of the blood unit.
		/// Implements BR_13.06
		/// Note: BR_13.05 is implemented in the Presentation Layer via a call to
		/// GetChildBloodUnitMessageString (see def above)
		/// </summary>
		/// <param name="comments">user comments stating reason for update</param>
		/// <param name="workloadProcessId">Workload Process to tie workload to</param>
		/// <param name="dtBCEMessages">BCE Messages</param>
		/// <returns>success/failure</returns>
		public bool RemoveFinalStatus(string comments, Common.WorkloadProcessID workloadProcessId, DataTable dtBCEMessages)
		{
			//CR 2941 Added dtBCEMessages
			return this.RemoveFinalStatus(comments, null, null, null, workloadProcessId, dtBCEMessages);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/3/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7269"> 
		///		<ExpectedInput>Valid pooled blood units whose target is invalidated</ExpectedInput>
		///		<ExpectedOutput>Success indicator of true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7270"> 
		///		<ExpectedInput>Pooled blood unit whose blood unit modification record row version value has changed</ExpectedInput>
		///		<ExpectedOutput>Row Version Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Rolls back final status (modified)
		/// Implements BR_13.03
		/// </summary>
		/// <param name="comments"></param>
		/// <param name="bloodUnitModification"></param>
		/// <param name="workloadProcessId"></param>
		/// <param name="dtBCEMessages">BCE Messages</param>
		/// <returns></returns>
		public bool RemoveFinalStatus(string comments, Common.WorkloadProcessID workloadProcessId, DataTable bloodUnitModification, DataTable dtBCEMessages)
		{
			//CR 2941 Added dtBCEMessages
			return this.RemoveFinalStatus(comments, null, null, bloodUnitModification, workloadProcessId, dtBCEMessages);
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3619"> 
		///		<ExpectedInput>UnitstatusCode, Comment, Patienttransfusion, WorkloadProcessID</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3652"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Rolls back final status (transfused)
		/// Implements BR_13.03, BR_13.05
		/// </summary>
		/// <param name="comments"></param>
		/// <param name="patientTransfusion"></param>
		/// <param name="workloadProcessId"></param>
		/// <param name="dtBCEMessages">BCE Messages</param>
		/// <returns></returns>
		public bool RemoveFinalStatus(string comments, BOL.PatientTransfusion patientTransfusion, Common.WorkloadProcessID workloadProcessId, DataTable dtBCEMessages)
		{
			//CR 2941 Added dtBCEMessages
			return this.RemoveFinalStatus(comments, patientTransfusion, null, null, workloadProcessId, dtBCEMessages);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6709"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>pass</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6710"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>rowversionsexception</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Rolls back final status (transferred)
		/// Implements BR_13.03
		/// </summary>
		/// <param name="comments"></param>
		/// <param name="outgoingShipment"></param>
		/// <param name="workloadProcessId"></param>
		/// <param name="dtBCEMessages">BCE Messages</param>
		/// <returns></returns>
		public bool RemoveFinalStatus(string comments, DataTable outgoingShipment, Common.WorkloadProcessID workloadProcessId, DataTable dtBCEMessages)
		{
			//CR 2941 Added dtBCEMessages
			return this.RemoveFinalStatus(comments, null, outgoingShipment, null, workloadProcessId, dtBCEMessages);
		}

		/// <summary>
		/// Rolls back final status (discarded, modified, transfused or transferred)
		/// mplements BR_13.03, BR_13.05, BR_13.06
		/// </summary>
		/// <param name="comments"></param>
		/// <param name="patientTransfusion"></param>
		/// <param name="outgoingShipment"></param>
		/// <param name="bloodUnitModification"></param>
		/// <param name="workloadProcessId"></param>
		/// <param name="dtBCEMessages">BCE Messages</param>
		/// <returns></returns>
		private bool RemoveFinalStatus(string comments, BOL.PatientTransfusion patientTransfusion, DataTable outgoingShipment, DataTable bloodUnitModification, Common.WorkloadProcessID workloadProcessId, DataTable dtBCEMessages)
		{
			//CR 2941 Added dtBCEMessages
			bool result = false;
			//Cr 3180
			System.Data.DataTable dtBloodUnitMediaRestoreRestiction = null;
			Guid [] bloodUnitGuids = new Guid[1];
			bloodUnitGuids[0] = this.BloodUnitGuid;
			DataTable dtWorkloadEvent = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, bloodUnitGuids, Common.WorkloadTransactionType.Unit, true);

			this.BloodUnitStatus.Comments = comments;
			this.BloodUnitStatus.PresumedTransfusedIndicator = false;
			this.BloodUnitStatus.DiscardCannedCommentGuid = Guid.Empty;
			System.Data.DataTable dtUnitStatus = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(true);
			dtUnitStatus.Rows.Add(this.BloodUnitStatus.LoadDataRowFromThis(dtUnitStatus.NewRow()));

			DataTable dtTransfusion = new DataTable();
			dtTransfusion.Columns.Add(TABLE.PatientTransfusion.PatientTransfusionGuid, typeof(System.Guid));
			dtTransfusion.Columns.Add(TABLE.PatientTransfusion.RecordStatusCode, typeof(char));
			dtTransfusion.Columns.Add(TABLE.PatientTransfusion.LastUpdateUser, typeof(string));
			dtTransfusion.Columns.Add(TABLE.PatientTransfusion.RowVersion, typeof(byte[]));

			if (patientTransfusion != null)
			{
				DataRow drTransfusion = dtTransfusion.NewRow();
				drTransfusion[TABLE.PatientTransfusion.PatientTransfusionGuid] = patientTransfusion.PatientTransfusionGuid;
				drTransfusion[TABLE.PatientTransfusion.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
				drTransfusion[TABLE.PatientTransfusion.LastUpdateUser] = Common.LogonUser.LogonUserName;
				drTransfusion[TABLE.PatientTransfusion.RowVersion] = patientTransfusion.RowVersion;

				dtTransfusion.Rows.Add(drTransfusion);
			}

			//When a final status of Transferred is removed, the Outgoing Shipment
			//record must be inactivated, otherwise it will not be
			//permitted to be added to any future invoices
			DataTable dtBloodUnitShipment = new DataTable();
			dtBloodUnitShipment.Columns.Add(VbecsTables.BloodUnitShipment.BloodUnitShipmentGuid, typeof(Guid));
			dtBloodUnitShipment.Columns.Add(VbecsTables.BloodUnitShipment.RecordStatusCode, typeof(char));
			dtBloodUnitShipment.Columns.Add(VbecsTables.BloodUnitShipment.RowVersion, typeof(byte[]));
			dtBloodUnitShipment.Columns.Add(VbecsTables.VamcDivision.DivisionCode, typeof(string));
			dtBloodUnitShipment.Columns.Add(VbecsTables.BloodUnitShipment.LastUpdateUser, typeof(string));

			if ((outgoingShipment != null)&&(outgoingShipment.Rows.Count>0))
			{
				DataRow dr = outgoingShipment.Rows[0];
				//
				DataRow drBloodUnitShipment = dtBloodUnitShipment.NewRow();
				drBloodUnitShipment[VbecsTables.BloodUnitShipment.BloodUnitShipmentGuid] = dr[VbecsTables.BloodUnitShipment.BloodUnitShipmentGuid];
				drBloodUnitShipment[VbecsTables.BloodUnitShipment.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
				drBloodUnitShipment[VbecsTables.BloodUnitShipment.RowVersion] = dr[VbecsTables.BloodUnitShipment.RowVersion];
				drBloodUnitShipment[VbecsTables.VamcDivision.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
				drBloodUnitShipment[VbecsTables.BloodUnitShipment.LastUpdateUser] = Common.LogonUser.LogonUserName;
				//
				dtBloodUnitShipment.Rows.Add(drBloodUnitShipment);
			}

			//When a final status of Modified is removed, the Blood Unit Modification
			//record(s) must be inactivated.
			System.Data.DataTable dtBloodUnitModification = new System.Data.DataTable(VbecsTables.BloodUnitModification.TableName);
			dtBloodUnitModification.Columns.Add(VbecsTables.BloodUnitModification.BloodUnitModificationGuid, typeof(System.Guid));
			dtBloodUnitModification.Columns.Add(VbecsTables.BloodUnitModification.RecordStatusCode, typeof(string));
			dtBloodUnitModification.Columns.Add(VbecsTables.BloodUnitModification.RowVersion, typeof(byte[]));	

			if ((bloodUnitModification != null)&&(bloodUnitModification.Rows.Count>0))
			{
				foreach(DataRow dr in bloodUnitModification.Rows)
				{
					DataRow drBloodUnitModification = dtBloodUnitModification.NewRow();
					//
					drBloodUnitModification[VbecsTables.BloodUnitModification.BloodUnitModificationGuid] = dr[VbecsTables.BloodUnitModification.BloodUnitModificationGuid];
					drBloodUnitModification[VbecsTables.BloodUnitModification.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
					drBloodUnitModification[VbecsTables.BloodUnitModification.RowVersion] = dr[VbecsTables.BloodUnitModification.RowVersion];
					//
					dtBloodUnitModification.Rows.Add(drBloodUnitModification);
				}
			}

			DataTable dtFinance = DAL.BloodUnitFinance.GetEmptyBloodUnitFinanceTableSchema(true);

			// BR_13.06 : When a final status of discard or transfer is inactivated, 
			// any return credit previously assigned to the blood unit is removed. 
			if (this.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Discarded ||
				this.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Transferred)
			{
				dtFinance.Rows.Add(this.BloodUnitFinance.LoadDataRowFromThis(dtFinance.NewRow()));
				//CR 3180
				if(this.BloodUnitStatus.UnitStatusCode == Common.UnitStatusCode.Transferred)
				{
					dtBloodUnitMediaRestoreRestiction = BOL.BloodUnitMedia.RestoreRestriction(this, Common.UpdateFunction.UC013FrmInvalidateFinalStatus);
				}
			}

			BOL.ExceptionReport exRep = new BOL.ExceptionReport(Common.ExceptionType.Final_unit_status_removed, comments, Common.LogonUser.LogonUserName);
			exRep.GenerateExceptionRemoveFinalUnitStatus(this,(Common.UnitStatusCode)(this.GetPreviousBloodUnitStatusTextAndCodeByGuid(this.BloodUnitGuid)[1]));
			DataTable dtException = exRep.ExceptionData.Table.Clone();
			dtException.Rows.Add(exRep.ExceptionData.ItemArray);

			//CR 3180
			result = (DAL.BloodUnit.RemoveFinalStatus(dtUnitStatus , dtBloodUnitMediaRestoreRestiction, dtFinance, Common.UpdateFunction.UC013FrmInvalidateFinalStatus, dtWorkloadEvent, dtTransfusion, dtBloodUnitShipment, dtBloodUnitModification, dtException, dtBCEMessages));
			return result;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7385"> 
		///		<ExpectedInput>Valid irradiated blood unit</ExpectedInput>
		///		<ExpectedOutput>Success indicator of true</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7386"> 
		///		<ExpectedInput>Irradiated blood unit whose row version value has changed</ExpectedInput>
		///		<ExpectedOutput>Row Version Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8138"> 
		///		<ExpectedInput>Invalid product modification code (Pooled)</ExpectedInput>
		///		<ExpectedOutput>Success indicator = false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// RemoveUnitModification
		/// </summary>
		/// <param name="bloodUnitModification"></param>
		/// <param name="comments"></param>
		/// <param name="workloadProcessId"></param>
		/// <param name="dtBCEMessages">BCE Messages</param>
		/// <returns></returns>
		public bool RemoveUnitModification(DataRow bloodUnitModification, string comments, ref Common.WorkloadProcessID workloadProcessId, DataTable dtBCEMessages)
		{
			//CR 2941 Added dtBCEMessages
			Common.ProductModification modification = Common.Utility.GetProductModificationCodeFromString(bloodUnitModification[VbecsTables.BloodUnitModification.ProductModificationCode].ToString());
			
			if (modification == Common.ProductModification.Deglycerolized)
			{
				workloadProcessId = Common.WorkloadProcessID.DeglycerolizeUnit;
			}
			else if	(modification == Common.ProductModification.Frozen)
			{
				workloadProcessId = Common.WorkloadProcessID.FreezeUnit;
			}
			else if (modification == Common.ProductModification.Irradiated)
			{
				workloadProcessId = Common.WorkloadProcessID.IrradiateUnit;
			}
			else if (modification == Common.ProductModification.LeukoReduced)
			{
				workloadProcessId = Common.WorkloadProcessID.LeukoreduceUnit;
			}
			else if	(modification == Common.ProductModification.Rejuvenated)
			{
				workloadProcessId = Common.WorkloadProcessID.RejuvenateUnit;
			}
			else if (modification == Common.ProductModification.Thawed)
			{
				workloadProcessId = Common.WorkloadProcessID.Thaw;
			}
			else if (modification == Common.ProductModification.VolumeReduced)
			{
				workloadProcessId = Common.WorkloadProcessID.VolumeReduce;
			}
			else if (modification == Common.ProductModification.Washed)
			{
				workloadProcessId = Common.WorkloadProcessID.WashUnit;
			}
			System.Guid [] bloodUnitGuids = new System.Guid [1];
			bloodUnitGuids[0] = (Guid)bloodUnitModification[VbecsTables.BloodUnitModification.FromBloodUnitGuid];

			if ((modification == Common.ProductModification.Deglycerolized)||
				(modification == Common.ProductModification.Frozen)||
				(modification == Common.ProductModification.Irradiated)||
				(modification == Common.ProductModification.LeukoReduced)||
				(modification == Common.ProductModification.Rejuvenated)||
				(modification == Common.ProductModification.Thawed)||
				(modification == Common.ProductModification.VolumeReduced)||
				(modification == Common.ProductModification.Washed))
			{
				System.Data.DataTable dtBloodUnitModification = new System.Data.DataTable(VbecsTables.BloodUnitModification.TableName);
				dtBloodUnitModification.Columns.Add(VbecsTables.BloodUnitModification.BloodUnitModificationGuid, typeof(System.Guid));
				dtBloodUnitModification.Columns.Add(VbecsTables.BloodUnitModification.RecordStatusCode, typeof(string));
				dtBloodUnitModification.Columns.Add(VbecsTables.BloodUnitModification.RowVersion, typeof(byte[]));	

				DataRow drBloodUnitModification = dtBloodUnitModification.NewRow();
				drBloodUnitModification[VbecsTables.BloodUnitModification.BloodUnitModificationGuid] = bloodUnitModification[VbecsTables.BloodUnitModification.BloodUnitModificationGuid];
				drBloodUnitModification[VbecsTables.BloodUnitModification.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
				drBloodUnitModification[VbecsTables.BloodUnitModification.RowVersion] = bloodUnitModification[VbecsTables.BloodUnitModification.RowVersion];
				dtBloodUnitModification.Rows.Add(drBloodUnitModification);
				//
				//"Rewind" the modified target unit
				System.Data.DataTable dtBloodUnit = DAL.BloodUnit.GetEmptyBloodUnitTableSchema(true);
				if ((this.BloodProductGuid != (Guid)bloodUnitModification[VbecsTables.BloodUnitModification.FromBloodUnitGuid])||
					(this.OriginalVolume != Convert.ToDecimal(bloodUnitModification[VbecsTables.BloodUnitModification.FromOriginalVolume])))
				{
					System.Guid bpGuid = (System.Guid)bloodUnitModification[VbecsTables.BloodUnitModification.FromBloodProductGuid];
					BOL.BloodProduct tmpBp = new BOL.BloodProduct(bpGuid);
 
					DataRow drBloodUnit = this.LoadDataRowFromThis(dtBloodUnit.NewRow());
					drBloodUnit[TABLE.BloodUnit.BloodUnitGuid] = bloodUnitModification[VbecsTables.BloodUnitModification.FromBloodUnitGuid];
					drBloodUnit[TABLE.BloodUnit.BloodProductGuid] = bloodUnitModification[VbecsTables.BloodUnitModification.FromBloodProductGuid];
					drBloodUnit[TABLE.BloodUnit.OriginalVolume] = bloodUnitModification[VbecsTables.BloodUnitModification.FromOriginalVolume];

					//Since we store the UnitProductCode along with BloodProductGuid , we have to rollback BOTH!
					//Normalize the DB!!  Performance be jiggered!
					drBloodUnit[TABLE.BloodUnit.UnitProductCode] = tmpBp.ProductCode;

					dtBloodUnit.Rows.Add(drBloodUnit);
				}
				//
				System.Data.DataTable dtBloodUnitMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(true);
				if (this.BloodUnitMedia.UnitExpirationDate != Convert.ToDateTime(bloodUnitModification[TABLE.BloodUnitModification.FromUnitExpirationDate]))
				{
					DataRow drBloodUnitMedia = this.BloodUnitMedia.LoadDataRowFromThis(dtBloodUnitMedia.NewRow());
					drBloodUnitMedia[TABLE.BloodUnitMedia.UnitExpirationDate] = bloodUnitModification[TABLE.BloodUnitModification.FromUnitExpirationDate];
					dtBloodUnitMedia.Rows.Add(drBloodUnitMedia);
				}

				//Rollback the ModificationCost
				System.Data.DataTable dtBloodUnitFinance = DAL.BloodUnitFinance.GetEmptyBloodUnitFinanceTableSchema(true);
				decimal modCost = Convert.ToDecimal(bloodUnitModification[TABLE.BloodUnitModification.ModificationCost]);
				if (modCost > decimal.Zero)
				{
					DataRow drBloodUnitFinance = this.BloodUnitFinance.LoadDataRowFromThis(dtBloodUnitFinance.NewRow());
					if ((this.BloodUnitFinance.UnitCost - modCost) > decimal.Zero)
					{
						drBloodUnitFinance[TABLE.BloodUnitFinance.UnitCost] = this.BloodUnitFinance.UnitCost - modCost;
					}
					else
					{
						//Instead of setting this to zero - we might want to grab a cost from the ProductShipper table
						//Although Kania says the likelihood of this code ever getting his is like ... very slim
						drBloodUnitFinance[TABLE.BloodUnitFinance.UnitCost] = decimal.Zero;
					}
					dtBloodUnitFinance.Rows.Add(drBloodUnitFinance);
				}

				BOL.ExceptionReport exRep = new BOL.ExceptionReport(Common.ExceptionType.Final_unit_status_removed, comments, Common.LogonUser.LogonUserName);
				exRep.GenerateExceptionRemoveUnitModification(this, bloodUnitModification);
				DataTable dtException = exRep.ExceptionData.Table.Clone();
				dtException.Rows.Add(exRep.ExceptionData.ItemArray);
				//
				DataTable dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, bloodUnitGuids, Common.WorkloadTransactionType.Unit, true);
				return DAL.BloodUnit.RemoveUnitModification(dtBloodUnit, dtBloodUnitModification, dtBloodUnitMedia, Common.UpdateFunction.UC013FrmInvalidateFinalStatus, dtException, dtBloodUnitFinance, dtWorkload, dtBCEMessages);
			}
			return false;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/23/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="4054"> 
		///		<ExpectedInput>techid, time, string, id</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4349"> 
		///		<ExpectedInput>invalid parameters</ExpectedInput>
		///		<ExpectedOutput>RowVersion excpetion</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Frees directed/crossmatched unit
		/// Implements BR_71.04
		/// </summary>
		/// <param name="updateTechId">last valid status code of this unit</param>
		/// <param name="updateTime">last valid status code of this unit</param>
		/// <param name="comments">user comments stating reason for update</param>
		/// <param name="workloadProcessId">Process Id for workload event</param>
		/// <returns>success/failure</returns>
		public bool FreeDirectedBloodUnit(string updateTechId, System.DateTime updateTime, string comments, Common.WorkloadProcessID workloadProcessId)
		{
			// Generate exception report
			ExceptionReport exRep = new ExceptionReport(Common.ExceptionType.Release_patient_restriction, comments, Common.LogonUser.LogonUserName);
			exRep.GenerateExceptionPatientRestriction(this);
			DataTable dtExceptionPatientRestriction = exRep.ExceptionData.Table.Clone();
			dtExceptionPatientRestriction.Rows.Add(exRep.ExceptionData.ItemArray);
			//
			DataTable dtBloodUnit = DAL.BloodUnit.GetEmptyBloodUnitTableSchema(true);
			dtBloodUnit.Columns.Add(TABLE.BloodUnit.LastUpdateDate, typeof(System.DateTime));
			dtBloodUnit.Columns.Add(TABLE.BloodUnit.LastUpdateUser, typeof(string));

			DataRow drBloodUnit = this.LoadDataRowFromThis(dtBloodUnit.NewRow());
			// BR_71.04 : The system changes the donation type of the unit to "V" (Voluntary Allogenic)
			drBloodUnit[TABLE.BloodUnit.DonationTypeId] = Common.DonationTypeId.VolunteerAllogeneicDonor;
			drBloodUnit[TABLE.BloodUnit.LastUpdateDate] = updateTime;
			drBloodUnit[TABLE.BloodUnit.LastUpdateUser] = updateTechId;
			dtBloodUnit.Rows.Add(drBloodUnit);
			//
			DataTable dtBloodUnitMedia = DAL.BloodUnitMedia.GetEmptyBloodUnitMediaTableSchema(true);

			DataRow drBloodUnitMedia = this.BloodUnitMedia.LoadDataRowFromThis(dtBloodUnitMedia.NewRow());
			drBloodUnitMedia[TABLE.BloodUnitMedia.Comments] = comments;
			drBloodUnitMedia[TABLE.BloodUnitMedia.DirectedPatientGuid] = DBNull.Value;
			drBloodUnitMedia[TABLE.BloodUnitMedia.RemoveRestrictionDate] = updateTime;
			drBloodUnitMedia[TABLE.BloodUnitMedia.RemoveRestrictionTechId] = updateTechId;
			dtBloodUnitMedia.Rows.Add(drBloodUnitMedia);
			//
			Guid [] bloodUnitGuids = new Guid[1];
			bloodUnitGuids[0] = BloodUnitGuid;
			DataTable dtWorkloadEvent = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
			dtWorkloadEvent = Common.Utility.AppendLastUpdateInformation(dtWorkloadEvent, Common.UpdateFunction.UC071FrmFreeDirectedUnit);

			return (DAL.BloodUnit.FreeDirectedUnit
				(
				dtBloodUnit, dtBloodUnitMedia, dtWorkloadEvent, dtExceptionPatientRestriction, Common.UpdateFunction.UC071FrmFreeDirectedUnit)
				);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7983"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7984"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetModifiedBloodUnits
		/// </summary>
		/// <returns></returns>
		public static System.Data.DataTable GetModifiedBloodUnits()
		{
			return DAL.BloodUnit.GetModifiedBloodUnits();
		}

		#endregion

		#region Implementation of IBaseBusinessObject

		///<Developers>
		///	<Developer>Greg "Maximus" Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5205"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5206"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LoadDataRowFromThis
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public new System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.BloodUnitGuid))
			{
				dtRow[TABLE.BloodUnit.BloodUnitGuid] = this.BloodUnitGuid;
			}
			
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.LabelTypeCode))
			{
				dtRow[TABLE.BloodUnit.LabelTypeCode] = this.BarcodeType;
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.EyeReadableUnitId))
			{
				dtRow[TABLE.BloodUnit.EyeReadableUnitId] = this.EyeReadableUnitId;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitMedia.DirectedPatientGuid))
			{
				dtRow[TABLE.BloodUnitMedia.DirectedPatientGuid] = this.DirectedPatient.PatientGuid;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.UnitProductCode))
			{
				dtRow[TABLE.BloodUnit.UnitProductCode] = this.ProductCode;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.DivisionCode))
			{
				dtRow[TABLE.BloodUnit.DivisionCode] = this.DivisionCode;		
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitMedia.UnitExpirationDate))
			{
				dtRow[TABLE.BloodUnitMedia.UnitExpirationDate] = this.BloodUnitMedia.UnitExpirationDate;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.UnitContainerLotNumber))
			{
				if (this.UnitContainerLotNumber != string.Empty)
					dtRow[TABLE.BloodUnit.UnitContainerLotNumber] = this.UnitContainerLotNumber;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.UnitDivisionId))
			{
				dtRow[TABLE.BloodUnit.UnitDivisionId] = this.UnitDivisionId;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.BloodProductGuid))
			{
				dtRow[TABLE.BloodUnit.BloodProductGuid] = this.BloodProductGuid;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.UnitInDate))
			{
				if (this.UnitInDate != System.DateTime.MinValue)
					dtRow[TABLE.BloodUnit.UnitInDate] = this.UnitInDate;
				else
					dtRow[TABLE.BloodUnit.UnitInDate] = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitMedia.BloodTypeCode))
			{
				dtRow[TABLE.BloodUnitMedia.BloodTypeCode] = this.BloodUnitMedia.BloodTypeCode;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.UnitFdaRegistrationNumber))
			{
				if (this.UnitFdaRegistrationNumber != string.Empty)
					dtRow[TABLE.BloodUnit.UnitFdaRegistrationNumber] = this.UnitFdaRegistrationNumber;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.LabelUnitId))
			{
				dtRow[TABLE.BloodUnit.LabelUnitId] = this.LabelUnitId;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodProduct.ProductTypeCode))
			{
				dtRow[TABLE.BloodProduct.ProductTypeCode] = this.ProductType.ProductTypeCode;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodProduct.ProductName))
			{
				dtRow[TABLE.BloodProduct.ProductName] = this.ProductName;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodProduct.ProductShortName))
			{
				dtRow[TABLE.BloodProduct.ProductShortName] = this.ProductShortName;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitStatus.BiohazardousWarningIndicator))
			{
				dtRow[TABLE.BloodUnitStatus.BiohazardousWarningIndicator] = this.BloodUnitStatus.BiohazardousWarningIndicator;
			}
			if (dtRow.Table.Columns.Contains(TABLE.BloodUnit.DonationTypeId))
			{
				dtRow[TABLE.BloodUnit.DonationTypeId]  = this.DonationType.DonationTypeId;
			}

			if(dtRow.Table.Columns.Contains(TABLE.DonationType.DonationTypeText))
			{
				dtRow[TABLE.DonationType.DonationTypeText] = this.DonationType.DonationTypeText;
			}
			if(dtRow.Table.Columns.Contains(TABLE.ProductType.ProductTypeName))
			{
				dtRow[TABLE.ProductType.ProductTypeName] = this.ProductType.ProductTypeText;
			}
	
			if(dtRow.Table.Columns.Contains(TABLE.BloodProduct.MaximumStorageTime))
			{
				dtRow[TABLE.BloodProduct.MaximumStorageTime] = (decimal)((this.MaximumStorageDays * 24)+(this.MaximumStorageHours));
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitStatus.UnitStatusComments))
			{
				if (this.BloodUnitStatus.UnitStatusComments != string.Empty)
					dtRow[TABLE.BloodUnitStatus.UnitStatusComments] = this.BloodUnitStatus.UnitStatusComments;					
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitStatus.DiscardCannedCommentGuid))
			{
				if (this.BloodUnitStatus.DiscardCannedCommentGuid != System.Guid.Empty)
					dtRow[TABLE.BloodUnitStatus.DiscardCannedCommentGuid] = this.BloodUnitStatus.DiscardCannedCommentGuid;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitStatusCodeCurrent.UnitStatusCode))
			{
				dtRow[TABLE.BloodUnitStatusCodeCurrent.UnitStatusCode] = Common.Utility.GetUnitStatusCodeCharFromEnum(this.BloodUnitStatus.UnitStatusCode);
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.RecordStatusCode))
			{
				dtRow[TABLE.BloodUnit.RecordStatusCode] = Common.Utility.GetRecordStatusCodeCharFromEnum(this.RecordStatusCode);
			}

			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitStatus.QuarantineIndicator))
			{
				dtRow[TABLE.BloodUnitStatus.QuarantineIndicator] = this.BloodUnitStatus.QuarantineIndicator;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitMedia.TestingStatusId))
			{
				if (this.BloodUnitMedia.TestingStatusId > 0)
					dtRow[TABLE.BloodUnitMedia.TestingStatusId] = this.BloodUnitMedia.TestingStatusId;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.OriginalVolume))
			{
				dtRow[TABLE.BloodUnit.OriginalVolume] = this.OriginalVolume;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.DiscardedPlasmaVolume))
			{
				dtRow[TABLE.BloodUnit.DiscardedPlasmaVolume] = this.DiscardedPlasmaVolume;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitMedia.Comments))
			{
				dtRow[TABLE.BloodUnitMedia.Comments] = this.BloodUnitMedia.Comments;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnitShipment.IncomingShipmentGuid))
			{
				dtRow[TABLE.BloodUnitShipment.IncomingShipmentGuid] = this.IncomingShipmentGuid;
			}
			if(dtRow.Table.Columns.Contains(TABLE.RhFactor.RhFactorCode))
			{
				dtRow[TABLE.RhFactor.RhFactorCode] = this.BloodUnitMedia.AboRh.RHFactorCode;
			}
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.ProcessingTechId))
			{
				dtRow[TABLE.BloodUnit.ProcessingTechId] = this.ProcessingTechId;
			}
			if (dtRow.Table.Columns.Contains(TABLE.BloodUnit.EntryMethodCode))
			{
				if (this.EntryMethodCode != string.Empty)
				{
					dtRow[TABLE.BloodUnit.EntryMethodCode] = this.EntryMethodCode;
				}
			}
			//Added row version columns for blood unit
			if(dtRow.Table.Columns.Contains(TABLE.BloodUnit.RowVersion))
			{
				dtRow.Table.Columns[TABLE.BloodUnit.RowVersion].ReadOnly = false;
				dtRow[TABLE.BloodUnit.RowVersion] = this.RowVersion;
			}

			return dtRow;
		}
		#endregion

		#region Implementation of ICloneable


		#endregion

	}
}